import { Component, inject } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { Router, RouterOutlet } from '@angular/router';

import { NgbNavConfig } from '@ng-bootstrap/ng-bootstrap';
import { LoadingBarRouterModule } from '@ngx-loading-bar/router';
import { TranslateService } from '@ngx-translate/core';
import { Actions, ofActionDispatched, Store } from '@ngxs/store';
import { forkJoin, Observable } from 'rxjs';

import { LogoutAction } from './shared/action/auth.action';
import { GetCountriesAction } from './shared/action/country.action';
import { GetSettingOptionAction } from './shared/action/setting.action';
import { GetStatesAction } from './shared/action/state.action';
import { IValues } from './shared/interface/setting.interface';
import { SettingState } from './shared/state/setting.state';

@Component({
  selector: 'app-root',
  imports: [RouterOutlet, LoadingBarRouterModule],
  templateUrl: './app.html',
  styleUrl: './app.scss',
})
export class App {
  private actions = inject(Actions);
  private router = inject(Router);
  private titleService = inject(Title);
  private store = inject(Store);
  private translate = inject(TranslateService);

  // constructor(){
  //   this.translate.use('en')
  // }

  setting$: Observable<IValues | null> = inject(Store).select(SettingState.setting);

  public favIcon: HTMLLinkElement | null;
  public googleMapApiKey: string | undefined;

  constructor() {
    const config = inject(NgbNavConfig);

    const settingCall$ = this.store.dispatch(new GetSettingOptionAction());

    forkJoin([settingCall$]).subscribe({
      complete: () => {
        this.store.dispatch(new GetCountriesAction());
        this.store.dispatch(new GetStatesAction());
        this.setting$.subscribe(setting => {
          this.googleMapApiKey = setting?.general.google_map_api_key;
          this.loadGoogleMapApi();
          // Set Direction
          const lang = this.store.selectSnapshot(state => state.language.localLanguage.locale);
          this.translate.use(lang ? lang : setting?.general.default_language.locale || 'en');

          if (
            setting?.general?.admin_site_language_direction === 'rtl' ||
            setting?.general.default_language.is_rtl
          ) {
            document.getElementsByTagName('html')[0].setAttribute('dir', 'rtl');
            document.body.classList.add('ltr');
          } else {
            document.getElementsByTagName('html')[0].removeAttribute('dir');
            document.body.classList.remove('ltr');
          }

          // Set Favicon
          this.favIcon = document.querySelector('#appIcon');
          if (
            this.favIcon &&
            setting &&
            setting.general &&
            setting.general.favicon_image &&
            setting.general.favicon_image.original_url
          ) {
            this.favIcon.href = <string>setting.general.favicon_image.original_url;
          }

          // Set site title
          this.titleService.setTitle(
            setting?.general?.site_title && setting?.general?.site_tagline
              ? `${setting?.general?.site_title} | ${setting?.general?.site_tagline}`
              : '',
          );
        });
      },
    });

    // customize default values of navs used by this component tree
    config.destroyOnHide = false;
    config.roles = false;

    this.actions.pipe(ofActionDispatched(LogoutAction)).subscribe(() => {
      void this.router.navigate(['/auth/login']);
    });
  }

  loadGoogleMapApi() {
    if (!document.getElementById('google-maps-script')) {
      const script = document.createElement('script');
      script.id = 'google-maps-script';
      if (this.googleMapApiKey) {
        script.src = `https://maps.googleapis.com/maps/api/js?key=${'YOUR_GOOGLE_API_KEY'}`;
      } else {
        script.src = `https://maps.googleapis.com/maps/api/js?key=`;
      }
      document.body.appendChild(script);
    }
  }
}
