import { AsyncPipe } from '@angular/common';
import { Component, inject, Renderer2, DOCUMENT } from '@angular/core';
import {
  AbstractControl,
  FormArray,
  FormBuilder,
  FormControl,
  FormGroup,
  ReactiveFormsModule,
} from '@angular/forms';

import {
  NgbAccordionBody,
  NgbAccordionButton,
  NgbAccordionCollapse,
  NgbAccordionDirective,
  NgbAccordionHeader,
  NgbAccordionItem,
  NgbAccordionToggle,
  NgbCollapse,
  NgbNav,
  NgbNavContent,
  NgbNavItem,
  NgbNavItemRole,
  NgbNavLink,
  NgbNavLinkBase,
  NgbNavOutlet,
} from '@ng-bootstrap/ng-bootstrap';
import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Select2, Select2Data, Select2Module, Select2SearchEvent } from 'ng-select2-component';
import { debounceTime, forkJoin, Observable, Subject } from 'rxjs';

import { GetBlogsAction } from '../../shared/action/blog.action';
import { GetBrandsAction } from '../../shared/action/brand.action';
import { GetCategoriesAction } from '../../shared/action/category.action';
import { GetCouponsAction } from '../../shared/action/coupon.action';
import { GetProductsAction } from '../../shared/action/product.action';
import {
  GetAppSettingOptionAction,
  UpdateAppSettingOptionAction,
} from '../../shared/action/setting.action';
import { GetStoresAction } from '../../shared/action/store.action';
import { PageWrapper } from '../../shared/components/page-wrapper/page-wrapper';
import { Button } from '../../shared/components/ui/button/button';
import { FormFields } from '../../shared/components/ui/form-fields/form-fields';
import { ImageUpload } from '../../shared/components/ui/image-upload/image-upload';
import { Link } from '../../shared/components/ui/link/link';
import * as data from '../../shared/data/home-page';
import * as media from '../../shared/data/media-config';
import { HasPermissionDirective } from '../../shared/directive/has-permission.directive';
import { Params } from '../../shared/interface/core.interface';
import { IAppSetting, IAppValues } from '../../shared/interface/setting.interface';
import { IBanners } from '../../shared/interface/theme.interface';
import { BlogState } from '../../shared/state/blog.state';
import { BrandState } from '../../shared/state/brand.state';
import { CategoryState } from '../../shared/state/category.state';
import { CouponState } from '../../shared/state/coupon.state';
import { ProductState } from '../../shared/state/product.state';
import { SettingState } from '../../shared/state/setting.state';
import { StoreState } from '../../shared/state/store.state';

export type CustomLinkForm = AbstractControl & {
  controls: { [key: string]: AbstractControl };
  registerControl: (name: string, control: AbstractControl) => void;
  addControl: (name: string, control: AbstractControl) => void;
  removeControl: (name: string) => void;
  setControl: (name: string, control: AbstractControl) => void;
};

@Component({
  selector: 'app-app-setting',
  templateUrl: './app-setting.html',
  styleUrl: './app-setting.scss',
  imports: [
    PageWrapper,
    ReactiveFormsModule,
    NgbNav,
    NgbNavItem,
    NgbNavItemRole,
    NgbNavLink,
    NgbNavLinkBase,
    NgbNavContent,
    Button,
    NgbAccordionDirective,
    NgbAccordionItem,
    NgbAccordionHeader,
    NgbAccordionToggle,
    NgbAccordionButton,
    NgbCollapse,
    NgbAccordionCollapse,
    NgbAccordionBody,
    FormFields,
    ImageUpload,
    Link,
    Select2Module,
    NgbNavOutlet,
    HasPermissionDirective,
    AsyncPipe,
    TranslateModule,
  ],
})
export class AppSetting {
  private store = inject(Store);
  private formBuilder = inject(FormBuilder);
  private renderer = inject(Renderer2);
  private document = inject<Document>(DOCUMENT);

  product$: Observable<Select2Data> = inject(Store).select(ProductState.products);
  coupons$: Observable<Select2Data> = inject(Store).select(CouponState.coupons);
  setting$: Observable<IAppValues | null> = inject(Store).select(SettingState.appSetting);
  categories$: Observable<Select2Data> = inject(Store).select(CategoryState.categories);
  brand$: Observable<Select2Data> = inject(Store).select(BrandState.brands);
  store$: Observable<Select2Data> = inject(Store).select(StoreState.stores);
  blogs$: Observable<Select2Data> = inject(Store).select(BlogState.blogs);

  public form: FormGroup;
  public page_data: IAppValues | null;
  public active = 'home_banner';
  public banner = 1;
  public sub_banner = 1;
  public main_content = 1;
  public sort = data.sort;
  public productIds: number[] = [];
  public mediaConfig = media.mediaConfig;

  private search = new Subject<string>();
  private destroy$ = new Subject<void>();

  public filter = {
    status: 1,
    search: '',
    paginate: 15,
    ids: '',
    with_union_products: 0,
    is_approved: 1,
  };

  constructor() {
    this.form = new FormGroup({
      values: new FormGroup({
        home_banner: new FormGroup({
          status: new FormControl(true),
          banners: new FormArray([]),
        }),
        categories_list: new FormGroup({
          status: new FormControl(true),
          title: new FormControl(),
          category_ids: new FormControl([]),
        }),
        offer_products: new FormGroup({
          status: new FormControl(true),
          title: new FormControl(),
          sub_title: new FormControl(),
          product_ids: new FormControl([]),
        }),
        section_1_products: new FormGroup({
          status: new FormControl(true),
          title: new FormControl(),
          sub_title: new FormControl(),
          product_ids: new FormControl([]),
        }),
        section_2_products: new FormGroup({
          status: new FormControl(true),
          title: new FormControl(),
          sub_title: new FormControl(),
          product_ids: new FormControl([]),
        }),
        coupons: new FormGroup({
          status: new FormControl(true),
          title: new FormControl(),
          sub_title: new FormControl(),
          coupon_ids: new FormControl([]),
        }),
        section_3_products: new FormGroup({
          status: new FormControl(true),
          title: new FormControl(),
          sub_title: new FormControl(),
          product_ids: new FormControl([]),
        }),
        seller: new FormGroup({
          status: new FormControl(true),
          title: new FormControl(),
          sub_title: new FormControl(),
          store_ids: new FormControl([]),
        }),
        brands: new FormGroup({
          status: new FormControl(true),
          title: new FormControl(),
          brand_ids: new FormControl([]),
        }),
        blogs: new FormGroup({
          status: new FormControl(true),
          title: new FormControl(),
          sub_title: new FormControl(),
          blog_ids: new FormControl([]),
        }),
        navigate_button: new FormGroup({
          status: new FormControl(true),
          title: new FormControl(),
          button_text: new FormControl(),
          redirect_link: new FormGroup({
            link: new FormControl(),
            link_type: new FormControl(),
          }),
        }),
        products_ids: new FormControl([]),
      }),
    });
  }

  get homeBanner(): FormArray {
    return this.form.get('values.home_banner.banners') as FormArray;
  }

  ngOnInit() {
    const blogs$ = this.store.dispatch(new GetBlogsAction({ status: 1 }));
    const app_setting$ = this.store.dispatch(new GetAppSettingOptionAction());
    const coupon$ = this.store.dispatch(new GetCouponsAction());
    const categories$ = this.store.dispatch(
      new GetCategoriesAction({ status: 1, type: 'product' }),
    );
    const brand$ = this.store.dispatch(new GetBrandsAction({ status: 1 }));
    const store$ = this.store.dispatch(new GetStoresAction({ status: 1 }));

    forkJoin([blogs$, app_setting$, categories$, coupon$, brand$, store$]).subscribe({
      complete: () => {
        this.store.select(SettingState.appSetting).subscribe({
          next: setting => {
            if (setting?.products_ids) {
              this.filter['paginate'] =
                setting?.products_ids?.length >= 15 ? setting?.products_ids?.length : 15;
              this.filter['ids'] = setting?.products_ids?.join();
              this.filter['with_union_products'] = setting?.products_ids?.length
                ? setting?.products_ids?.length >= 15
                  ? 0
                  : 1
                : 0;
            }
            this.store.dispatch(new GetProductsAction(this.filter)).subscribe({
              complete: () => {
                this.patchForm();
              },
            });
          },
        });
      },
    });

    this.search
      .pipe(debounceTime(300)) // Adjust the debounce time as needed (in milliseconds)
      .subscribe(inputValue => {
        this.filter['search'] = inputValue;
        this.getProducts(this.filter);
        this.renderer.addClass(this.document.body, 'loader-none');
      });
  }

  patchForm() {
    this.store.select(SettingState.appSetting).subscribe(setting => {
      this.page_data = setting;
      this.form.patchValue({
        values: {
          home_banner: {
            status: setting?.home_banner?.status,
          },
          categories_list: {
            status: setting?.categories_list?.status,
            title: setting?.categories_list?.title,
            category_ids: setting?.categories_list?.category_ids,
          },
          offer_products: {
            status: setting?.offer_products?.status,
            title: setting?.offer_products?.title,
            sub_title: setting?.offer_products?.sub_title,
            product_ids: setting?.offer_products?.product_ids,
          },
          section_1_products: {
            status: setting?.section_1_products?.status,
            title: setting?.section_1_products?.title,
            sub_title: setting?.section_1_products?.sub_title,
            product_ids: setting?.section_1_products?.product_ids,
          },
          section_2_products: {
            status: setting?.section_2_products?.status,
            title: setting?.section_2_products?.title,
            sub_title: setting?.section_2_products?.sub_title,
            product_ids: setting?.section_2_products?.product_ids,
          },
          coupons: {
            status: setting?.coupons?.status,
            title: setting?.coupons?.title,
            sub_title: setting?.coupons?.sub_title,
            coupon_ids: setting?.coupons?.coupon_ids,
          },
          section_3_products: {
            status: setting?.section_3_products?.status,
            title: setting?.section_3_products?.title,
            sub_title: setting?.section_3_products?.sub_title,
            product_ids: setting?.section_3_products?.product_ids,
          },
          seller: {
            status: setting?.seller?.status,
            title: setting?.seller?.title,
            sub_title: setting?.seller?.sub_title,
            store_ids: setting?.seller?.store_ids,
          },
          brands: {
            status: setting?.brands?.status,
            title: setting?.brands?.title,
            brand_ids: setting?.brands?.brand_ids,
          },
          blogs: {
            status: setting?.blogs?.status,
            title: setting?.blogs?.title,
            sub_title: setting?.blogs?.sub_title,
            blog_ids: setting?.blogs?.blog_ids,
          },
          navigate_button: {
            status: setting?.navigate_button?.status,
            title: setting?.navigate_button?.title,
            button_text: setting?.navigate_button?.button_text,
            redirect_link: {
              link: setting?.navigate_button?.redirect_link?.link,
              link_type: setting?.navigate_button?.redirect_link?.link_type,
              product_ids: setting?.navigate_button?.redirect_link?.product_ids,
            },
          },
          products_ids: setting?.products_ids,
        },
      });

      this.homeBanner.clear();
      setting?.home_banner?.banners?.forEach((banner: IBanners) =>
        this.homeBanner.push(
          this.formBuilder.group({
            redirect_link: new FormGroup({
              link: new FormControl(banner?.redirect_link?.link),
              link_type: new FormControl(banner?.redirect_link?.link_type),
            }),
            status: new FormControl(banner?.status),
            image_url: new FormControl(banner?.image_url),
          }),
        ),
      );
    });
  }

  getProducts(filter: Params) {
    this.filter['search'] = filter['search'];
    this.filter['ids'] = this.filter['search'].length
      ? ''
      : (this.page_data?.products_ids?.join() ?? '');
    // this.filter['paginate'] = this.page_data?.products_ids?.length >= 15 ? this.page_data?.products_ids?.length : 15;
    this.filter['paginate'] =
      this.page_data?.products_ids?.length && this.page_data?.products_ids?.length >= 15
        ? this.page_data.products_ids.length
        : 15;
    this.store.dispatch(new GetProductsAction(this.filter));
    this.renderer.addClass(this.document.body, 'loader-none');
  }

  productDropdown(event: Select2) {
    if (event['innerSearchText']) {
      this.search.next('');
      this.getProducts(this.filter);
    }
  }

  searchProduct(event: Select2SearchEvent) {
    this.search.next(event.search);
  }

  addBanner(event: Event) {
    event.preventDefault();
    this.homeBanner.push(
      this.formBuilder.group({
        redirect_link: new FormGroup({
          link: new FormControl(''),
          link_type: new FormControl(''),
          product_ids: new FormControl(''),
        }),
        image_url: new FormControl(),
        status: new FormControl(true),
      }),
    );
  }

  remove(index: number) {
    if (this.homeBanner.length <= 1) return;
    this.homeBanner.removeAt(index);
  }

  selectImage(url: string, key: string) {
    this.form.get(key)?.setValue(url ? url : null);
  }

  selectBannerArray(url: string, index: number) {
    this.homeBanner
      .at(index)
      .get('image_url')
      ?.setValue(url ? url : null);
  }

  // Merge Products Ids
  concatDynamicProductKeys(obj: IAppSetting): number[] {
    const result: number[] = [];

    function traverse(value: unknown): void {
      if (Array.isArray(value)) {
        // look inside array
        value.forEach(item => traverse(item));
      } else if (value && typeof value === 'object') {
        for (const [key, val] of Object.entries(value)) {
          if (key === 'product_ids' && Array.isArray(val)) {
            result.push(...val);
          } else if (val && typeof val === 'object') {
            traverse(val);
          }
        }
      }
    }

    traverse(obj);
    return result;
  }

  submit() {
    const productIds = Array.from(new Set(this.concatDynamicProductKeys(this.form.value)));
    this.form.get('values.products_ids')?.setValue(productIds);

    if (this.form.valid) {
      this.store.dispatch(new UpdateAppSettingOptionAction(this.form.value));
    }
  }

  ngOnDestroy() {
    this.destroy$.next();
    this.destroy$.complete();
  }
}
