import { Component, PLATFORM_ID, inject, input } from '@angular/core';
import {
  FormBuilder,
  FormControl,
  FormGroup,
  Validators,
  FormArray,
  ReactiveFormsModule,
} from '@angular/forms';
import { Router, ActivatedRoute } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Select2Data, Select2Module } from 'ng-select2-component';
import { Subject, of } from 'rxjs';
import { switchMap, mergeMap, takeUntil } from 'rxjs/operators';

import {
  CreateAttributeAction,
  EditAttributeAction,
  UpdateAttributeAction,
} from '../../../shared/action/attribute.action';
import { EditFormLanguageAction } from '../../../shared/action/language.action';
import { Button } from '../../../shared/components/ui/button/button';
import { FormFields } from '../../../shared/components/ui/form-fields/form-fields';
import { LanguageRedirection } from '../../../shared/components/ui/language-redirection/language-redirection';
import { IAttribute } from '../../../shared/interface/attribute.interface';
import { AttributeState } from '../../../shared/state/attribute.state';

@Component({
  selector: 'app-form-attribute',
  templateUrl: './form-attribute.html',
  styleUrls: ['./form-attribute.scss'],
  imports: [
    LanguageRedirection,
    ReactiveFormsModule,
    FormFields,
    Select2Module,
    Button,
    TranslateModule,
  ],
})
export class FormAttribute {
  private store = inject(Store);
  private route = inject(ActivatedRoute);
  private router = inject(Router);
  private formBuilder = inject(FormBuilder);
  private platformId = inject<Object>(PLATFORM_ID);

  readonly type = input<string>();

  public form: FormGroup;
  public documents = [{ value: '', hex_color: '', id: '' }];
  public id: number;
  public lang: string;
  public attribute: IAttribute;
  private destroy$ = new Subject<void>();

  public variantStyle: Select2Data = [
    {
      value: 'rectangle',
      label: 'Rectangle',
    },
    {
      value: 'circle',
      label: 'Circle',
    },
    {
      value: 'radio',
      label: 'Radio',
    },
    {
      value: 'dropdown',
      label: 'Dropdown',
    },
    {
      value: 'image',
      label: 'Image',
    },
    {
      value: 'color',
      label: 'Color',
    },
  ];

  constructor() {
    this.form = this.formBuilder.group({
      name: new FormControl('', [Validators.required]),
      style: new FormControl('rectangle', [Validators.required]),
      status: new FormControl(1),
      value: this.formBuilder.array([], [Validators.required]),
    });
  }

  get valueControl(): FormArray {
    return this.form.get('value') as FormArray;
  }

  ngOnInit() {
    this.route.params
      .pipe(
        switchMap(params => {
          if (!params['id']) return of();

          if (params['lang']) this.lang = params['lang'];
          this.store.dispatch(new EditFormLanguageAction(params['lang']));

          return this.store
            .dispatch(new EditAttributeAction(params['id']))
            .pipe(mergeMap(() => this.store.select(AttributeState.selectedAttribute)));
        }),
        takeUntil(this.destroy$),
      )
      .subscribe(attribute => {
        this.id = attribute?.id!;
        this.attribute = attribute!;
        // Set Value in form
        this.form.patchValue({
          name: attribute?.name,
          style: attribute?.style,
          status: attribute?.status,
        });
        // Set Attribute Values
        attribute?.attribute_values!.forEach(document =>
          this.valueControl.push(
            this.formBuilder.group({
              value: new FormControl(document.value, [Validators.required]),
              hex_color: new FormControl(document.hex_color),
              id: new FormControl(document.id, []),
            }),
          ),
        );
      });

    if (this.type() == 'create') {
      this.documents.forEach(document =>
        this.valueControl.push(
          this.formBuilder.group({
            value: [document.value, [Validators.required]],
            hex_color: [document.hex_color],
            id: [document.id],
          }),
        ),
      );
    }
  }

  add(event: Event) {
    event.preventDefault();
    const valueGroup =
      this.form.get('style')!.value === 'color'
        ? this.formBuilder.group({
            value: ['', [Validators.required]],
            hex_color: [''],
          })
        : this.formBuilder.group({
            value: ['', [Validators.required]],
          });
    this.valueControl.push(valueGroup);
  }

  remove(index: number) {
    if (this.valueControl.length <= 1) return;
    this.valueControl.removeAt(+index);
  }

  submit() {
    this.form.markAllAsTouched();
    let action = new CreateAttributeAction(this.form.value);

    if (this.type() == 'edit' && this.id) {
      action = new UpdateAttributeAction(this.form.value, this.id);
    }

    if (this.form.valid) {
      this.store.dispatch(action).subscribe({
        complete: () => {
          void this.router.navigateByUrl('/attribute');
        },
      });
    }
  }

  onSlugInput(event: Event): void {
    const inputElement = event.target as HTMLInputElement;
    const updatedValue = inputElement.value.replace(/\s+/g, '-'); // Replace spaces with hyphens
    this.form.get('slug')?.setValue(updatedValue); // Update the form control value
  }

  ngOnDestroy() {
    this.destroy$.next();
    this.destroy$.complete();
  }
}
