import { Component, inject } from '@angular/core';
import {
  FormBuilder,
  FormControl,
  FormGroup,
  FormsModule,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { Router, RouterModule } from '@angular/router';

import { TranslatePipe } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Observable } from 'rxjs';

import { ForgotPassWordAction } from '../../../shared/action/auth.action';
import { Alert } from '../../../shared/components/ui/alert/alert';
import { Button } from '../../../shared/components/ui/button/button';
import { IValues } from '../../../shared/interface/setting.interface';
import { SettingState } from '../../../shared/state/setting.state';

@Component({
  selector: 'app-forgot-password',
  imports: [RouterModule, FormsModule, ReactiveFormsModule, TranslatePipe, Alert, Button],
  templateUrl: './forgot-password.html',
  styleUrls: ['./forgot-password.scss'],
})
export class ForgotPassword {
  private store = inject(Store);
  router = inject(Router);
  formBuilder = inject(FormBuilder);

  public form: FormGroup;
  public reCaptcha: boolean = true;

  setting$: Observable<IValues | null> = inject(Store).select(SettingState.setting);

  constructor() {
    this.form = this.formBuilder.group({
      email: ['', [Validators.required, Validators.email]],
      recaptcha: ['', [Validators.required]],
    });
    this.setting$.subscribe(setting => {
      if (
        (setting?.google_reCaptcha && !setting?.google_reCaptcha.status) ||
        !setting?.google_reCaptcha
      ) {
        this.form.removeControl('recaptcha');
        this.reCaptcha = false;
      } else {
        this.form.setControl('recaptcha', new FormControl(null, Validators.required));
        this.reCaptcha = true;
      }
    });
  }

  submit() {
    this.form.markAllAsTouched();
    if (this.form.valid) {
      this.store.dispatch(new ForgotPassWordAction(this.form.value)).subscribe({
        complete: () => {
          void this.router.navigateByUrl('/auth/otp');
        },
      });
    }
  }
}
