import { CommonModule } from '@angular/common';
import { Component, inject, PLATFORM_ID } from '@angular/core';
import {
  FormBuilder,
  FormControl,
  FormGroup,
  FormsModule,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { Router, RouterModule } from '@angular/router';

import { TranslatePipe } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Select2Data, Select2Module, Select2UpdateEvent } from 'ng-select2-component';
import { Observable } from 'rxjs';
import { map } from 'rxjs/operators';

import { CreateStoreAction } from '../../../shared/action/store.action';
import { Alert } from '../../../shared/components/ui/alert/alert';
import { Button } from '../../../shared/components/ui/button/button';
import * as data from '../../../shared/data/country-code';
import { IValues } from '../../../shared/interface/setting.interface';
import { IStores } from '../../../shared/interface/store.interface';
import { NotificationService } from '../../../shared/services/notification.service';
import { CountryState } from '../../../shared/state/country.state';
import { SettingState } from '../../../shared/state/setting.state';
import { StateState } from '../../../shared/state/state.state';
import { StoreState } from '../../../shared/state/store.state';
import { CustomValidators } from '../../../shared/validator/password-match';

@Component({
  selector: 'app-register',
  imports: [
    CommonModule,
    TranslatePipe,
    RouterModule,
    Button,
    FormsModule,
    ReactiveFormsModule,
    Select2Module,
    Alert,
  ],
  templateUrl: './register.html',
  styleUrls: ['./register.scss'],
})
export class Register {
  private store = inject(Store);
  private router = inject(Router);
  private notificationService = inject(NotificationService);
  private formBuilder = inject(FormBuilder);
  private platformId = inject<Object>(PLATFORM_ID);

  public form: FormGroup;
  public codes = data.countryCodes;

  public reCaptcha: boolean = true;

  setting$: Observable<IValues | null> = inject(Store).select(SettingState.setting);
  countries$: Observable<Select2Data> = inject(Store).select(CountryState.countries);
  store$: Observable<IStores | null> = inject(Store).select(StoreState.selectedStore);

  public states$: Observable<Select2Data>;

  constructor() {
    this.form = this.formBuilder.group(
      {
        store_name: new FormControl('', [Validators.required]),
        description: new FormControl('', [Validators.required]),
        country_id: new FormControl('', [Validators.required]),
        state_id: new FormControl('', [Validators.required]),
        city: new FormControl('', [Validators.required]),
        address: new FormControl('', [Validators.required]),
        pincode: new FormControl('', [Validators.required]),
        name: new FormControl('', [Validators.required]),
        email: new FormControl('', [Validators.required, Validators.email]),
        phone: new FormControl('', [Validators.required]),
        country_code: new FormControl('91', [Validators.required]),
        password: new FormControl('', [Validators.required]),
        password_confirmation: new FormControl('', [Validators.required]),
        status: new FormControl(1),
        recaptcha: new FormControl(null, Validators.required),
      },
      {
        validator: CustomValidators.MatchValidator('password', 'password_confirmation'),
      },
    );

    this.setting$.subscribe(setting => {
      if (
        (setting?.google_reCaptcha && !setting?.google_reCaptcha.status) ||
        !setting?.google_reCaptcha
      ) {
        this.form.removeControl('recaptcha');
        this.reCaptcha = false;
      } else {
        this.form.setControl('recaptcha', new FormControl(null, Validators.required));
        this.reCaptcha = true;
      }
    });
  }

  get passwordMatchError() {
    return this.form.getError('mismatch') && this.form.get('password_confirmation')?.touched;
  }

  countryChange(data: Select2UpdateEvent) {
    if (data && data?.value) {
      this.states$ = this.store
        .select(StateState.states)
        .pipe(map(filterFn => filterFn(+data?.value)));
      this.form.controls['state_id'].setValue('');
    } else {
      this.form.controls['state_id'].setValue('');
    }
  }

  submit() {
    this.form.markAllAsTouched();
    this.notificationService.notification = false;
    let action = new CreateStoreAction(this.form.value);
    if (this.form.valid) {
      this.store.dispatch(action).subscribe({
        complete: () => {
          void this.router.navigateByUrl('/login');
        },
      });
    }
  }
}
