import { Component, inject } from '@angular/core';
import { Router, RouterLink } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Observable } from 'rxjs';

import {
  DeleteAllAuthorAction,
  DeleteAuthorAction,
  GetAuthorsAction,
  UpdateAuthorStatusAction,
} from '../../shared/action/author.action';
import { PageWrapper } from '../../shared/components/page-wrapper/page-wrapper';
import { Table } from '../../shared/components/ui/table/table';
import { HasPermissionDirective } from '../../shared/directive/has-permission.directive';
import { IAuthor, IAuthorModel } from '../../shared/interface/author.interface';
import { Params } from '../../shared/interface/core.interface';
import { ITableClickedAction, ITableConfig } from '../../shared/interface/table.interface';
import { AuthorState } from '../../shared/state/author.state';

@Component({
  selector: 'app-author',
  imports: [PageWrapper, HasPermissionDirective, RouterLink, Table, TranslateModule],
  templateUrl: './author.html',
  styleUrl: './author.scss',
})
export class Author {
  private store = inject(Store);
  router = inject(Router);

  author$: Observable<IAuthorModel> = inject(Store).select(AuthorState.author);

  public tableConfig: ITableConfig = {
    columns: [
      { title: 'Name', dataField: 'author_name', sortable: true, sort_direction: 'desc' },
      {
        title: 'created_at',
        dataField: 'created_at',
        type: 'date',
        sortable: true,
        sort_direction: 'desc',
      },
      { title: 'status', dataField: 'status', type: 'switch' },
    ],
    rowActions: [
      { label: 'Edit', actionToPerform: 'edit', icon: 'ri-pencil-line', permission: 'brand.edit' },
      {
        label: 'Delete',
        actionToPerform: 'delete',
        icon: 'ri-delete-bin-line',
        permission: 'brand.destroy',
      },
    ],
    search: true,
    data: [] as IAuthor[],
    total: 0,
  };

  ngOnInit() {
    this.author$.subscribe(author => {
      this.tableConfig.data = author ? author?.data : [];
      this.tableConfig.total = author ? author?.total : 0;
    });
  }

  onTableChange(data?: Params) {
    this.store.dispatch(new GetAuthorsAction(data));
  }

  onActionClicked(action: ITableClickedAction) {
    if (action.actionToPerform == 'edit') this.edit(action.data);
    else if (action.actionToPerform == 'status') this.status(action.data);
    else if (action.actionToPerform == 'delete') this.delete(action.data);
    else if (action.actionToPerform == 'deleteAll') this.deleteAll(action.data);
  }

  edit(data: IAuthor) {
    void this.router.navigateByUrl(
      `/author/${this.store.selectSnapshot(state => state.language.localLanguage.locale)}/edit/${data.id}`,
    );
  }

  status(data: IAuthor) {
    this.store.dispatch(new UpdateAuthorStatusAction(data.id, data.status));
  }

  delete(data: IAuthor) {
    this.store.dispatch(new DeleteAuthorAction(data.id));
  }

  deleteAll(ids: number[]) {
    this.store.dispatch(new DeleteAllAuthorAction(ids));
  }
}
