import { AsyncPipe } from '@angular/common';
import { Component, inject, input } from '@angular/core';
import {
  FormBuilder,
  FormControl,
  FormGroup,
  FormsModule,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { ActivatedRoute, Router } from '@angular/router';

import {
  NgbDate,
  NgbDateParserFormatter,
  NgbDatepickerModule,
  NgbDateStruct,
  NgbInputDatepicker,
} from '@ng-bootstrap/ng-bootstrap';
import { TranslatePipe } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Select2Data, Select2Module, Select2UpdateEvent } from 'ng-select2-component';
import { map, mergeMap, Observable, of, Subject, switchMap, takeUntil } from 'rxjs';

import {
  CreateAuthorAction,
  EditAuthorAction,
  UpdateAuthorAction,
} from '../../../shared/action/author.action';
import { EditFormLanguageAction } from '../../../shared/action/language.action';
import { Button } from '../../../shared/components/ui/button/button';
import { FormFields } from '../../../shared/components/ui/form-fields/form-fields';
import { ImageUpload } from '../../../shared/components/ui/image-upload/image-upload';
import { LanguageRedirection } from '../../../shared/components/ui/language-redirection/language-redirection';
import * as data from '../../../shared/data/media-config';
import { IAttachment } from '../../../shared/interface/attachment.interface';
import { IAuthor } from '../../../shared/interface/author.interface';
import { IStores } from '../../../shared/interface/store.interface';
import { AuthorState } from '../../../shared/state/author.state';
import { CountryState } from '../../../shared/state/country.state';
import { StateState } from '../../../shared/state/state.state';
import { StoreState } from '../../../shared/state/store.state';

function convertToNgbDate(date: NgbDateStruct): NgbDate {
  return new NgbDate(date.year, date.month, date.day);
}

@Component({
  selector: 'app-form-author',
  imports: [
    Button,
    FormsModule,
    ReactiveFormsModule,
    NgbDatepickerModule,
    NgbInputDatepicker,
    TranslatePipe,
    FormFields,
    ImageUpload,
    AsyncPipe,
    Select2Module,
    LanguageRedirection,
  ],
  templateUrl: './form-author.html',
  styleUrl: './form-author.scss',
})
export class FormAuthor {
  private store = inject(Store);
  private route = inject(ActivatedRoute);
  private router = inject(Router);
  formatter = inject(NgbDateParserFormatter);
  private formBuilder = inject(FormBuilder);

  readonly type = input<string>();
  author$: Observable<IAuthor> = inject(Store).select(
    AuthorState.selectedAuthor,
  ) as Observable<IAuthor>;
  store$: Observable<IStores> = inject(Store).select(
    StoreState.selectedStore,
  ) as Observable<IStores>;
  countries$: Observable<Select2Data> = inject(Store).select(CountryState.countries);

  public form: FormGroup;
  public id: number;
  public lang: string;
  private destroy$ = new Subject<void>();
  public mediaConfig: data.IMediaConfig = data.mediaConfig;
  public birthDate: NgbDate | null;
  public deathDate: NgbDate | null;
  public states$: Observable<Select2Data>;
  public maxDate: { year: number; month: number; day: number };

  constructor() {
    const today = new Date();
    this.maxDate = {
      year: today.getFullYear(),
      month: today.getMonth() + 1,
      day: today.getDate(),
    };

    this.form = this.formBuilder.group({
      author_name: new FormControl('', [Validators.required]),
      slug: new FormControl(''),
      bio: new FormControl('', [Validators.required]),
      author_image_id: new FormControl([Validators.required]),
      author_cover_image_id: new FormControl(),
      country_id: new FormControl('', [Validators.required]),
      state_id: new FormControl('', [Validators.required]),
      city: new FormControl('', [Validators.required]),
      birth_date: new FormControl(''),
      death_date: new FormControl(''),
      languages: new FormControl(''),
      facebook: new FormControl(''),
      twitter: new FormControl(''),
      instagram: new FormControl(''),
      youtube: new FormControl(''),
      pinterest: new FormControl(''),
      status: new FormControl(1),
    });
  }

  ngOnInit() {
    this.route.params
      .pipe(
        switchMap(params => {
          if (!params['id']) return of();
          this.states$ = this.store.select(StateState.states).pipe(map(filterFn => filterFn(null)));
          if (params['lang']) this.lang = params['lang'];
          this.store.dispatch(new EditFormLanguageAction(params['lang']));
          return this.store
            .dispatch(new EditAuthorAction(params['id']))
            .pipe(mergeMap(() => this.store.select(AuthorState.selectedAuthor)));
        }),
        takeUntil(this.destroy$),
      )
      .subscribe(author => {
        this.id = author?.id!;

        this.birthDate = author?.birth_date
          ? convertToNgbDate(this.formatter.parse(author?.birth_date)!)
          : null;
        this.deathDate = author?.death_date
          ? convertToNgbDate(this.formatter.parse(author?.death_date)!)
          : null;

        this.form.patchValue({
          author_name: author?.author_name,
          slug: author?.slug,
          bio: author?.bio,
          author_image_id: author?.author_image_id,
          author_cover_image_id: author?.author_cover_image_id,
          country_id: author?.country_id,
          state_id: author?.state_id,
          city: author?.city,
          birth_date: author?.birth_date,
          death_date: author?.death_date,
          languages: author?.languages,
          facebook: author?.facebook,
          twitter: author?.twitter,
          instagram: author?.instagram,
          youtube: author?.youtube,
          pinterest: author?.pinterest,
          status: author?.status,
        });
      });
  }

  countryChange(data: Select2UpdateEvent) {
    if (data && data?.value) {
      this.states$ = this.store
        .select(StateState.states)
        .pipe(map(filterFn => filterFn(+data?.value)));

      if (data.component.focused) this.form.controls['state_id'].setValue('');
    } else {
      this.form.controls['state_id'].setValue('');
    }
  }

  selectAuthorImage(data: IAttachment) {
    if (!Array.isArray(data)) {
      this.form.controls['author_image_id'].setValue(data ? data.id : null);
    }
  }

  selectAuthorCoverImage(data: IAttachment) {
    if (!Array.isArray(data)) {
      this.form.controls['author_cover_image_id'].setValue(data ? data.id : null);
    }
  }

  onDateSelection(date: NgbDate) {
    if (!this.birthDate && !this.deathDate) {
      this.birthDate = date;
    } else if (this.birthDate && !this.deathDate && date && date.after(this.birthDate)) {
      this.deathDate = date;
    } else {
      this.deathDate = null;
      this.birthDate = date;
    }

    if (this.birthDate)
      this.form.controls['birth_date'].setValue(
        `${this.birthDate.year}-${this.birthDate.month}-${this.birthDate.day}`,
      );
    if (this.deathDate)
      this.form.controls['death_date'].setValue(
        `${this.deathDate?.year}-${this.deathDate?.month}-${this.deathDate?.day}`,
      );
  }

  submit() {
    this.form.markAllAsTouched();
    let action = new CreateAuthorAction(this.form.value);

    if (this.type() == 'edit' && this.id) {
      action = new UpdateAuthorAction(this.form.value, this.id);
    }

    if (this.form.valid) {
      this.store.dispatch(action).subscribe({
        complete: () => {
          void this.router.navigateByUrl('/author');
          this.form.reset();
        },
      });
    }
  }

  ngOnDestroy() {
    this.destroy$.next();
    this.destroy$.complete();
    this.form.controls['author_image_id'].setValue(null);
    this.form.controls['author_cover_image_id'].setValue(null);
    this.form.reset();
  }
}
