import { AsyncPipe } from '@angular/common';
import { Component, inject, input } from '@angular/core';
import {
  FormGroup,
  FormBuilder,
  FormControl,
  Validators,
  ReactiveFormsModule,
} from '@angular/forms';
import { ActivatedRoute, Router } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Select2Data, Select2Module } from 'ng-select2-component';
import { Observable, Subject, of } from 'rxjs';
import { switchMap, mergeMap, takeUntil } from 'rxjs/operators';

import {
  CreateCategoryAction,
  EditCategoryAction,
  UpdateCategoryAction,
} from '../../../shared/action/category.action';
import { EditFormLanguageAction } from '../../../shared/action/language.action';
import { GetZonesAction } from '../../../shared/action/zone.action';
import { PageWrapper } from '../../../shared/components/page-wrapper/page-wrapper';
import { AdvancedDropdown } from '../../../shared/components/ui/advanced-dropdown/advanced-dropdown';
import { Button } from '../../../shared/components/ui/button/button';
import { FormFields } from '../../../shared/components/ui/form-fields/form-fields';
import { ImageUpload } from '../../../shared/components/ui/image-upload/image-upload';
import { LanguageRedirection } from '../../../shared/components/ui/language-redirection/language-redirection';
import * as media from '../../../shared/data/media-config';
import { NumberDirective } from '../../../shared/directive/numbers-only.directive';
import { IAttachment } from '../../../shared/interface/attachment.interface';
import { ICategory } from '../../../shared/interface/category.interface';
import { IValues } from '../../../shared/interface/setting.interface';
import { CategoryState } from '../../../shared/state/category.state';
import { SettingState } from '../../../shared/state/setting.state';
import { ZoneState } from '../../../shared/state/zone.state';

@Component({
  selector: 'app-form-category',
  templateUrl: './form-category.html',
  styleUrls: ['./form-category.scss'],
  imports: [
    PageWrapper,
    LanguageRedirection,
    ReactiveFormsModule,
    FormFields,
    NumberDirective,
    AdvancedDropdown,
    ImageUpload,
    Button,
    TranslateModule,
    AsyncPipe,
    Select2Module,
  ],
})
export class FormCategory {
  private store = inject(Store);
  private route = inject(ActivatedRoute);
  private router = inject(Router);
  private formBuilder = inject(FormBuilder);

  readonly type = input<string>();
  readonly categories = input<ICategory[]>();
  readonly categoryType = input<string | null>('product');

  public form: FormGroup;
  public category: ICategory;
  public id: number;
  public mediaConfig = media.mediaConfig;
  public zones$: Observable<Select2Data> = inject(Store).select(ZoneState.zones);
  public setting$: Observable<IValues | null> = inject(Store).select(SettingState.setting);

  private destroy$ = new Subject<void>();

  constructor() {
    this.store.dispatch(new GetZonesAction());
    this.form = this.formBuilder.group({
      name: new FormControl('', [Validators.required]),
      slug: new FormControl(''),
      description: new FormControl(),
      parent_id: new FormControl(),
      type: new FormControl(this.categoryType(), []),
      commission_rate: new FormControl(),
      category_image_id: new FormControl(),
      category_icon_id: new FormControl(),
      category_meta_image_id: new FormControl(),
      meta_title: new FormControl(),
      meta_description: new FormControl(),
      is_allow_all_zone: new FormControl(1),
      exclude_zone_ids: new FormControl([]),
      zone_ids: new FormControl([]),
      status: new FormControl(true),
    });
  }

  ngOnChanges() {
    this.form.controls['type'].setValue(this.categoryType());
  }

  ngOnInit() {
    this.route.params
      .pipe(
        switchMap(params => {
          if (params['id']) this.id = params['id'];
          this.store.dispatch(new EditFormLanguageAction(params['lang']));

          if (!params['id']) return of();
          return this.store
            .dispatch(new EditCategoryAction(params['id']))
            .pipe(mergeMap(() => this.store.select(CategoryState.selectedCategory)));
        }),
        takeUntil(this.destroy$),
      )
      .subscribe(category => {
        this.category = category!;
        this.form.patchValue({
          name: this.category?.name,
          slug: this.category?.slug,
          description: this.category?.description,
          parent_id: this.category?.parent_id,
          type: this.category?.type,
          commission_rate: this.category?.commission_rate,
          category_image_id: this.category?.category_image_id,
          category_icon_id: this.category?.category_icon_id,
          category_meta_image_id: this.category?.category_meta_image_id,
          meta_title: this.category?.meta_title,
          meta_description: this.category?.meta_description,
          is_allow_all_zone: this.category?.is_allow_all_zone,
          exclude_zone_ids: this.category?.exclude_zones?.map(item => item.id),
          zone_ids: this.category?.zones?.map(item => item.id),
          status: this.category?.status,
        });
      });

    this.form.controls['is_allow_all_zone'].valueChanges.subscribe(data => {
      if (data === true || data === 1) {
        this.form.setControl('exclude_zone_ids', new FormControl([]));
        this.form.setControl('zone_ids', new FormControl([]));
      } else {
        this.form.setControl('zone_ids', new FormControl([], [Validators.required]));
        this.form.setControl('exclude_zone_ids', new FormControl([]));
      }
    });
  }

  selectItem(data: number[]) {
    if (Array.isArray(data) && data.length) {
      this.form.controls['parent_id'].setValue(data[0]);
    } else {
      this.form.controls['parent_id'].setValue('');
    }
  }

  selectCategoryImage(data: IAttachment) {
    if (!Array.isArray(data)) {
      this.form.controls['category_image_id'].setValue(data ? data.id : '');
    }
  }

  selectCategoryIcon(data: IAttachment) {
    if (!Array.isArray(data)) {
      this.form.controls['category_icon_id'].setValue(data ? data.id : '');
    }
  }

  selectMetaImage(data: IAttachment) {
    if (!Array.isArray(data)) {
      this.form.controls['category_meta_image_id'].setValue(data ? data.id : '');
    }
  }

  submit() {
    this.form.markAllAsTouched();
    let action = new CreateCategoryAction(this.form.value);

    if (this.type() == 'edit' && this.category?.id) {
      action = new UpdateCategoryAction(this.form.value, this.category.id);
    }

    if (this.form.valid) {
      this.store.dispatch(action).subscribe({
        complete: () => {
          if (this.type() == 'create') {
            this.form.reset();
            this.form.controls['category_image_id'].setValue('');
            this.form.controls['category_icon_id'].setValue('');
            this.form.controls['status'].setValue(true);
          } else {
            if (this.form.value.type === 'product') {
              void this.router.navigateByUrl('/category');
            } else {
              void this.router.navigateByUrl('/blog/category');
            }
          }
        },
      });
    }
  }

  ngOnDestroy() {
    this.destroy$.next();
    this.destroy$.complete();
  }
}
