import { AsyncPipe, KeyValuePipe } from '@angular/common';
import { Component, inject, Renderer2, DOCUMENT } from '@angular/core';
import { FormControl, FormGroup, ReactiveFormsModule } from '@angular/forms';
import { ActivatedRoute, Router } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Select2Data, Select2UpdateEvent, Select2Module } from 'ng-select2-component';
import { debounceTime, distinctUntilChanged, Observable } from 'rxjs';

import {
  EditFormLanguageAction,
  GetLanguageFilesAction,
  GetLanguageFilesFieldsAction,
  UpdateLanguageFilesFieldsAction,
} from '../../../shared/action/language.action';
import { PageWrapper } from '../../../shared/components/page-wrapper/page-wrapper';
import { Button } from '../../../shared/components/ui/button/button';
import { FormFields } from '../../../shared/components/ui/form-fields/form-fields';
import { Pagination } from '../../../shared/components/ui/pagination/pagination';
import { Params } from '../../../shared/interface/core.interface';
import { ILanguageFiledModel, ILanguageModel } from '../../../shared/interface/language.interface';
import { LanguageState } from '../../../shared/state/language.state';

@Component({
  selector: 'app-form-edit-file',
  templateUrl: './form-edit-file.html',
  styleUrl: './form-edit-file.scss',
  imports: [
    PageWrapper,
    FormFields,
    Select2Module,
    ReactiveFormsModule,
    Button,
    Pagination,
    AsyncPipe,
    KeyValuePipe,
    TranslateModule,
  ],
})
export class FormEditFile {
  private store = inject(Store);
  private route = inject(ActivatedRoute);
  private router = inject(Router);
  private renderer = inject(Renderer2);
  private document = inject<Document>(DOCUMENT);

  languageFiles$: Observable<Select2Data> = inject(Store).select(
    LanguageState.languagesFiles,
  ) as Observable<Select2Data>;
  languageFilesFields$: Observable<ILanguageFiledModel> = inject(Store).select(
    LanguageState.languagesFilesFields,
  );
  language$: Observable<ILanguageModel> = inject(Store).select(LanguageState.language);

  public defaultFileName: string;
  public form = new FormGroup({});
  public selectedFile: string;
  public totalItems: number;
  public id: number;
  public filter = {
    page: 1, // current page number
    paginate: 15, // Display per page,
    pagination: true,
    search: '',
  };
  public term = new FormControl();
  private isInitialized = false; // flag to prevent duplicate API call on initialization

  constructor() {
    this.term.valueChanges
      .pipe(debounceTime(400), distinctUntilChanged())
      .subscribe((data: string) => {
        this.filter.page = 1;
        this.filter.search = data;
        this.getFields(this.filter);
        this.renderer.addClass(this.document.body, 'loader-none');
      });
  }

  ngOnInit() {
    this.store.dispatch(new GetLanguageFilesAction()).subscribe({
      complete: () => {
        this.languageFiles$.subscribe((res: any) => {
          if (!this.isInitialized && res && res[0]) {
            this.selectedFile = String(res[0].value);
            this.getFields(this.filter); // Load fields only once on initialization
            this.isInitialized = true; // Set flag to true to avoid duplicate calls
          }
        });

        this.languageFilesFields$.subscribe(res => {
          this.totalItems = res.total;
          for (const key of Object.keys(res.data)) {
            // Add the control only if it doesn't exist
            if (!this.form.contains(key)) {
              this.form.addControl(key, new FormControl(''));
            }
          }
          this.form.patchValue(res.data); // Populate the form with field data
        });
      },
    });

    this.language$.subscribe(language => {
      if (language && language.data.length) {
        this.id = +(this.route.snapshot.paramMap.get('id') || 0);
        const matchedLanguage = language.data.find(lang => lang.id === this.id);

        if (matchedLanguage) {
          this.store.dispatch(new EditFormLanguageAction(matchedLanguage.locale));
        } else {
          console.warn('No matching language found for ID:', this.id);
        }
      }
    });
  }

  getFields(filter: Params) {
    this.store.dispatch(
      new GetLanguageFilesFieldsAction(this.selectedFile, { ...filter, id: this.id }),
    );
  }

  changeFile(data: Select2UpdateEvent) {
    if (data && data.value && this.isInitialized) {
      // Check isInitialized to avoid calling on initial load
      this.form.reset();
      this.selectedFile = String(data.value);
      this.getFields(this.filter);
    }
  }

  setPaginate(data: number) {
    Object.keys(this.form.controls).forEach(controlName => {
      this.form.removeControl(controlName);
    });

    this.filter.page = data;
    this.getFields(this.filter);
  }

  update() {
    this.store.dispatch(
      new UpdateLanguageFilesFieldsAction(this.selectedFile, this.form.value, this.filter),
    );
  }

  ngOnDestroy() {
    this.renderer.removeClass(this.document.body, 'loader-none');
  }
}
