import { Component, inject, input } from '@angular/core';
import {
  FormBuilder,
  FormControl,
  FormGroup,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { ActivatedRoute, Router } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Select2Data, Select2Module, Select2SearchEvent } from 'ng-select2-component';
import { debounceTime, mergeMap, of, Subject, switchMap, takeUntil } from 'rxjs';

import {
  CreateLanguageAction,
  EditLanguageAction,
  UpdateLanguageAction,
} from '../../../shared/action/language.action';
import { Button } from '../../../shared/components/ui/button/button';
import { FormFields } from '../../../shared/components/ui/form-fields/form-fields';
import { ILanguage } from '../../../shared/interface/language.interface';
import { LanguageState } from '../../../shared/state/language.state';

@Component({
  selector: 'app-form-language',
  templateUrl: './form-language.html',
  styleUrl: './form-language.scss',
  imports: [ReactiveFormsModule, FormFields, Select2Module, Button, TranslateModule],
})
export class FormLanguage {
  private store = inject(Store);
  private router = inject(Router);
  private route = inject(ActivatedRoute);
  private formBuilder = inject(FormBuilder);

  readonly type = input<string>();

  public form: FormGroup;
  public language: ILanguage | null;

  private destroy$ = new Subject<void>();
  private search = new Subject<string>();
  private countryFlagSearch = new Subject<string>();

  public localOptions: Select2Data = [
    { id: '1', label: 'English (En)', value: 'en' },
    { id: '2', label: 'Arabic (Ar)', value: 'ar' },
    { id: '3', label: 'Spanish (Es)', value: 'es' },
    { id: '4', label: 'French (Fr)', value: 'fr' },
    { id: '5', label: 'German (De)', value: 'de' },
    { id: '6', label: 'Chinese Simplified (Zh-CN)', value: 'zh-CN' },
    { id: '7', label: 'Chinese Traditional (Zh-TW)', value: 'zh-TW' },
    { id: '8', label: 'Japanese (Ja)', value: 'ja' },
    { id: '9', label: 'Korean (Ko)', value: 'ko' },
    { id: '10', label: 'Russian (Ru)', value: 'ru' },
    { id: '11', label: 'Portuguese (Pt)', value: 'pt' },
    { id: '12', label: 'Italian (It)', value: 'it' },
    { id: '13', label: 'Dutch (Nl)', value: 'nl' },
    { id: '14', label: 'Swedish (Sv)', value: 'sv' },
    { id: '15', label: 'Norwegian (No)', value: 'no' },
    { id: '16', label: 'Danish (Da)', value: 'da' },
    { id: '17', label: 'Finnish (Fi)', value: 'fi' },
    { id: '18', label: 'Turkish (Tr)', value: 'tr' },
    { id: '19', label: 'Greek (El)', value: 'el' },
    { id: '20', label: 'Hebrew (He)', value: 'he' },
    { id: '21', label: 'Hindi (Hi)', value: 'hi' },
    { id: '22', label: 'Thai (Th)', value: 'th' },
    { id: '23', label: 'Polish (Pl)', value: 'pl' },
  ];

  public countryFlags: Select2Data = [
    { id: '1', label: `<div class="iti-flag us"></div>`, value: 'us' },
    { id: '2', label: `<div class="iti-flag gb"></div>`, value: 'gb' },
    { id: '3', label: `<div class="iti-flag fr"></div>`, value: 'fr' },
    { id: '4', label: `<div class="iti-flag de"></div>`, value: 'de' },
    { id: '5', label: `<div class="iti-flag it"></div>`, value: 'it' },
    { id: '6', label: `<div class="iti-flag jp"></div>`, value: 'jp' },
    { id: '7', label: `<div class="iti-flag kr"></div>`, value: 'kr' },
    { id: '8', label: `<div class="iti-flag cn"></div>`, value: 'cn' },
    { id: '9', label: `<div class="iti-flag ca"></div>`, value: 'ca' },
    { id: '10', label: `<div class="iti-flag in"></div>`, value: 'in' },
    { id: '11', label: `<div class="iti-flag au"></div>`, value: 'au' },
    { id: '12', label: `<div class="iti-flag br"></div>`, value: 'br' },
    { id: '13', label: `<div class="iti-flag ru"></div>`, value: 'ru' },
    { id: '14', label: `<div class="iti-flag es"></div>`, value: 'es' },
    { id: '15', label: `<div class="iti-flag sa"></div>`, value: 'sa' },
    { id: '16', label: `<div class="iti-flag za"></div>`, value: 'za' },
    { id: '17', label: `<div class="iti-flag se"></div>`, value: 'se' },
    { id: '18', label: `<div class="iti-flag tr"></div>`, value: 'tr' },
    { id: '19', label: `<div class="iti-flag nl"></div>`, value: 'nl' },
    { id: '20', label: `<div class="iti-flag ch"></div>`, value: 'ch' },
    { id: '21', label: `<div class="iti-flag pt"></div>`, value: 'pt' },
    { id: '22', label: `<div class="iti-flag gr"></div>`, value: 'gr' },
    { id: '23', label: `<div class="iti-flag il"></div>`, value: 'il' },
    { id: '24', label: `<div class="iti-flag th"></div>`, value: 'th' },
    { id: '25', label: `<div class="iti-flag mx"></div>`, value: 'mx' },
    { id: '26', label: `<div class="iti-flag no"></div>`, value: 'no' },
    { id: '27', label: `<div class="iti-flag fi"></div>`, value: 'fi' },
    { id: '28', label: `<div class="iti-flag dk"></div>`, value: 'dk' },
    { id: '29', label: `<div class="iti-flag pl"></div>`, value: 'pl' },
    { id: '30', label: `<div class="iti-flag be"></div>`, value: 'be' },
  ];

  public filteredOptions: Select2Data = [...this.localOptions];
  public filteredFlagOptions: Select2Data = [...this.countryFlags];

  constructor() {
    this.form = this.formBuilder.group({
      name: new FormControl('', [Validators.required]),
      // flag: new FormControl('', [Validators.required]),
      locale: new FormControl('', [Validators.required]),
      is_rtl: new FormControl(false),
      status: new FormControl(true),
    });
  }

  ngOnInit() {
    this.route.params
      .pipe(
        switchMap(params => {
          if (!params['id']) return of();
          return this.store
            .dispatch(new EditLanguageAction(params['id']))
            .pipe(mergeMap(() => this.store.select(LanguageState.selectedLanguage)));
        }),
        takeUntil(this.destroy$),
      )
      .subscribe(language => {
        this.language = language;

        this.form.patchValue({
          name: this.language?.name,
          // flag: this.language?.flag,
          locale: this.language?.locale,
          is_rtl: this.language?.is_rtl,
          status: this.language?.status,
        });
      });

    this.search
      .pipe(debounceTime(100)) // Adjust the debounce time as needed (in milliseconds)
      .subscribe(inputValue => {
        this.filteredOptions = this.localOptions.filter(option =>
          inputValue
            .toLowerCase()
            .split(' ')
            .every(term => option.label.toLowerCase().includes(term)),
        );
      });

    this.countryFlagSearch
      .pipe(debounceTime(100)) // Adjust the debounce time as needed (in milliseconds)
      .subscribe(inputValue => {
        this.filteredFlagOptions = this.countryFlags.filter(option =>
          inputValue
            .toLowerCase()
            .split(' ')
            .every(term => option.label.toLowerCase().includes(term)),
        );
      });
  }

  searchDropdown(event: any) {
    if (event['innerSearchText']) {
      this.search.next(''); // Reset search if dropdown is closed
    }
  }

  searchData(event: Select2SearchEvent) {
    this.search.next(event.search); // Trigger search on typing
  }

  searchFlagDropdown(event: any) {
    if (event['innerSearchText']) {
      this.countryFlagSearch.next(''); // Reset search if dropdown is closed
    }
  }

  searchFlagData(event: Select2SearchEvent) {
    this.countryFlagSearch.next(event.search); // Trigger search on typing
  }

  submit() {
    this.form.markAllAsTouched();
    let action = new CreateLanguageAction(this.form.value);

    if (this.type() == 'edit' && this.language?.id) {
      action = new UpdateLanguageAction(this.form.value, this.language.id);
    }

    if (this.form.valid) {
      this.store.dispatch(action).subscribe({
        complete: () => {
          void this.router.navigateByUrl('/language');
        },
      });
    }
  }
}
