import { Component, inject } from '@angular/core';
import { Router, RouterLink } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Observable } from 'rxjs';

import {
  DeleteLanguageAction,
  GetLanguageAction,
  UpdateLanguageStatusAction,
} from '../../shared/action/language.action';
import { PageWrapper } from '../../shared/components/page-wrapper/page-wrapper';
import { Table } from '../../shared/components/ui/table/table';
import { HasPermissionDirective } from '../../shared/directive/has-permission.directive';
import { Params } from '../../shared/interface/core.interface';
import { ILanguage, ILanguageModel } from '../../shared/interface/language.interface';
import { ITableClickedAction, ITableConfig } from '../../shared/interface/table.interface';
import { LanguageState } from '../../shared/state/language.state';

@Component({
  selector: 'app-language',
  templateUrl: './language.html',
  styleUrl: './language.scss',
  imports: [PageWrapper, HasPermissionDirective, RouterLink, Table, TranslateModule],
})
export class Language {
  private store = inject(Store);
  router = inject(Router);

  language$: Observable<ILanguageModel> = inject(Store).select(LanguageState.language);

  public tableConfig: ITableConfig = {
    columns: [
      { title: 'name', dataField: 'name', sortable: true, sort_direction: 'desc' },
      { title: 'locale', dataField: 'locale', sortable: true, sort_direction: 'desc' },
      { title: 'status', dataField: 'status', type: 'switch' },
      {
        title: 'created_at',
        dataField: 'created_at',
        type: 'date',
        sortable: true,
        sort_direction: 'desc',
      },
    ],
    rowActions: [
      {
        label: 'Edit',
        actionToPerform: 'edit',
        icon: 'ri-pencil-line',
        permission: 'language.edit',
      },
      {
        label: 'File Edit',
        actionToPerform: 'editFile',
        icon: 'ri-global-line',
        permission: 'language.edit',
      },
      {
        label: 'Delete',
        actionToPerform: 'delete',
        icon: 'ri-delete-bin-line',
        permission: 'language.destroy',
      },
    ],
    search: true,
    data: [] as ILanguage[],
    total: 0,
  };

  ngOnInit() {
    this.language$.subscribe(language => {
      this.tableConfig.data = language ? language?.data : [];
      this.tableConfig.total = language ? language?.total : 0;
    });
  }

  onTableChange(data?: Params) {
    this.store.dispatch(new GetLanguageAction(data));
  }

  onActionClicked(action: ITableClickedAction) {
    if (action.actionToPerform == 'edit') this.edit(action.data);
    else if (action.actionToPerform == 'status') this.status(action.data);
    else if (action.actionToPerform == 'delete') this.delete(action.data);
    else if (action.actionToPerform == 'editFile') this.editFile(action.data);
    // else if(action.actionToPerform == 'deleteAll')
    //   this.deleteAll(action.data)
  }

  edit(data: ILanguage) {
    void this.router.navigateByUrl(`/language/edit/${data.id}`);
  }

  editFile(data: ILanguage) {
    void this.router.navigateByUrl(`/language/edit-file/${data.id}`);
  }

  status(data: ILanguage) {
    this.store.dispatch(new UpdateLanguageStatusAction(data.id, data.status));
  }

  delete(data: ILanguage) {
    this.store.dispatch(new DeleteLanguageAction(data.id));
  }
}
