import { AsyncPipe, DatePipe, NgClass, TitleCasePipe, UpperCasePipe } from '@angular/common';
import { Component, inject, viewChild } from '@angular/core';
import { ActivatedRoute, RouterLink } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Select2Data, Select2Module, Select2UpdateEvent } from 'ng-select2-component';
import { Observable, Subject, of } from 'rxjs';
import { mergeMap, switchMap, takeUntil } from 'rxjs/operators';

import { ShippingNoteModal } from './shipping-note-modal/shipping-note-modal';
import { GetOrderStatusAction } from '../../../shared/action/order-status.action';
import { DownloadInvoiceAction, ViewOrderAction } from '../../../shared/action/order.action';
import { PageWrapper } from '../../../shared/components/page-wrapper/page-wrapper';
import { IOrderStatus, IOrderStatusModel } from '../../../shared/interface/order-status.interface';
import { IOrder, IOrderStatusActivities } from '../../../shared/interface/order.interface';
import { CurrencySymbolPipe } from '../../../shared/pipe/currency-symbol.pipe';
import { OrderStatusState } from '../../../shared/state/order-status.state';
import { OrderState } from '../../../shared/state/order.state';
import { PosInvoiceModal } from '../checkout/modal/pos-invoice-modal/pos-invoice-modal';

@Component({
  selector: 'app-details',
  templateUrl: './details.html',
  styleUrls: ['./details.scss'],
  imports: [
    NgClass,
    PageWrapper,
    Select2Module,
    RouterLink,
    PosInvoiceModal,
    ShippingNoteModal,
    AsyncPipe,
    UpperCasePipe,
    TitleCasePipe,
    DatePipe,
    TranslateModule,
    CurrencySymbolPipe,
  ],
  providers: [CurrencySymbolPipe],
})
export class Details {
  private store = inject(Store);
  private datePipe = inject(DatePipe);
  private route = inject(ActivatedRoute);

  orderStatus$: Observable<IOrderStatusModel> = inject(Store).select(OrderStatusState.orderStatus);
  orderStatuses$: Observable<Select2Data> = inject(Store).select(
    OrderStatusState.orderStatuses,
  ) as Observable<Select2Data>;

  readonly PosInvoice = viewChild<PosInvoiceModal>('posInvoice');
  readonly ShippingNote = viewChild<ShippingNoteModal>('shippingNote');

  public order: IOrder;
  public statuses: IOrderStatus[] = [];
  public init: boolean;
  private destroy$ = new Subject<void>();

  constructor() {
    this.store.dispatch(new GetOrderStatusAction());
    this.init = true;
  }

  ngOnInit() {
    this.route.params
      .pipe(
        switchMap(params => {
          if (!params['id']) return of();
          return this.store
            .dispatch(new ViewOrderAction(params['id']))
            .pipe(mergeMap(() => this.store.select(OrderState.selectedOrder)));
        }),
        takeUntil(this.destroy$),
      )
      .subscribe(order => {
        this.order = order!;
        if (this.order && this.order?.order_status_activities) {
          this.order?.order_status_activities?.map(actStatus => {
            this.orderStatus$.subscribe(res => {
              res.data.map(status => {
                if (actStatus.status == status.name) {
                  let convertDate = this.datePipe.transform(
                    actStatus?.changed_at,
                    'dd MMM yyyy hh:mm:a',
                  )!;
                  status['activities_date'] = convertDate;
                }
              });
            });
          });
        }
      });
  }

  updateOrderStatus(data: Select2UpdateEvent) {
    if (data && data?.value) {
      if (!this.init) void this.ShippingNote()?.openModal(this.order?.id, Number(data.value));

      this.init = false;
    }
  }

  getDate(status: string, order: IOrderStatusActivities[]) {
    const getdate = order?.map(res => {
      const convertDate = this.datePipe.transform(res?.changed_at, 'dd MMM yyyy hh:mm:a')!;
      return status === res.status ? convertDate : ' ';
    });
    return getdate;
  }

  ngOnDestroy() {
    this.statuses = [];
    this.destroy$.next();
    this.destroy$.complete();
  }

  download(id: number) {
    this.store.dispatch(new DownloadInvoiceAction({ order_number: id }));
  }
}
