import { AsyncPipe, NgClass } from '@angular/common';
import { Component, inject, input } from '@angular/core';
import {
  FormBuilder,
  FormControl,
  FormGroup,
  FormsModule,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { ActivatedRoute, Router } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Editor, NgxEditorModule } from 'ngx-editor';
import { Observable, Subject, of } from 'rxjs';
import { mergeMap, switchMap, takeUntil } from 'rxjs/operators';

import { EditFormLanguageAction } from '../../../shared/action/language.action';
import {
  CreatePageAction,
  EditPageAction,
  UpdatePageAction,
} from '../../../shared/action/page.action';
import { Button } from '../../../shared/components/ui/button/button';
import { FormFields } from '../../../shared/components/ui/form-fields/form-fields';
import { ImageUpload } from '../../../shared/components/ui/image-upload/image-upload';
import { LanguageRedirection } from '../../../shared/components/ui/language-redirection/language-redirection';
import * as media from '../../../shared/data/media-config';
import { IAttachment } from '../../../shared/interface/attachment.interface';
import { IPage } from '../../../shared/interface/page.interface';
import { PageState } from '../../../shared/state/page.state';

@Component({
  selector: 'app-form-page',
  templateUrl: './form-page.html',
  styleUrls: ['./form-page.scss'],
  imports: [
    LanguageRedirection,
    ReactiveFormsModule,
    FormFields,
    FormsModule,
    NgClass,
    NgxEditorModule,
    ImageUpload,
    Button,
    AsyncPipe,
    TranslateModule,
  ],
})
export class FormPage {
  private store = inject(Store);
  private route = inject(ActivatedRoute);
  private router = inject(Router);
  private formBuilder = inject(FormBuilder);

  readonly type = input<string>();

  page$: Observable<IPage> = inject(Store).select(PageState.selectedPage) as Observable<IPage>;

  private destroy$ = new Subject<void>();
  public html = '';
  public form: FormGroup;
  public id: number;
  public mediaConfig = media.mediaConfig;
  public editor: Editor;
  public isCodeEditor = true;
  public textArea = new FormControl('');
  public lang: string;

  constructor() {
    this.form = this.formBuilder.group({
      title: new FormControl('', [Validators.required]),
      slug: new FormControl(''),
      content: new FormControl(),
      meta_title: new FormControl(),
      meta_description: new FormControl(),
      page_meta_image_id: new FormControl(),
      status: new FormControl(1),
    });
  }

  ngOnInit() {
    this.route.params
      .pipe(
        switchMap(params => {
          if (!params['id']) return of();

          if (params['lang']) this.lang = params['lang'];
          this.store.dispatch(new EditFormLanguageAction(params['lang']));

          return this.store
            .dispatch(new EditPageAction(params['id']))
            .pipe(mergeMap(() => this.store.select(PageState.selectedPage)));
        }),
        takeUntil(this.destroy$),
      )
      .subscribe(page => {
        this.id = page?.id!;
        this.form.patchValue({
          title: page?.title,
          content: page?.content,
          meta_title: page?.meta_title,
          meta_description: page?.meta_description,
          status: page?.status,
        });
      });

    this.editor = new Editor();
  }

  selectMetaImage(data: IAttachment) {
    if (!Array.isArray(data)) {
      this.form.controls['page_meta_image_id'].setValue(data ? data.id : null);
    }
  }

  getText(_event: KeyboardEvent) {
    this.form.controls['content'].setValue(this.textArea.value);
  }

  getData(_description: string) {
    this.textArea.setValue(this.html);
  }

  submit() {
    this.form.markAllAsTouched();
    let action = new CreatePageAction(this.form.value);

    if (this.type() == 'edit' && this.id) {
      action = new UpdatePageAction(this.form.value, this.id);
    }

    if (this.form.valid) {
      this.store.dispatch(action).subscribe({
        complete: () => {
          void this.router.navigateByUrl('/page');
        },
      });
    }
  }

  ngOnDestroy() {
    this.destroy$.next();
    this.destroy$.complete();
  }
}
