import { Component, inject } from '@angular/core';
import { Router, RouterLink } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Observable } from 'rxjs';

import {
  DeleteAllPageAction,
  DeletePageAction,
  GetPagesAction,
  UpdatePageStatusAction,
} from '../../shared/action/page.action';
import { PageWrapper } from '../../shared/components/page-wrapper/page-wrapper';
import { Table } from '../../shared/components/ui/table/table';
import { HasPermissionDirective } from '../../shared/directive/has-permission.directive';
import { Params } from '../../shared/interface/core.interface';
import { IPage, IPageModel } from '../../shared/interface/page.interface';
import { IValues } from '../../shared/interface/setting.interface';
import { ITableClickedAction, ITableConfig } from '../../shared/interface/table.interface';
import { PageState } from '../../shared/state/page.state';
import { SettingState } from '../../shared/state/setting.state';

@Component({
  selector: 'app-page',
  templateUrl: './page.html',
  styleUrls: ['./page.scss'],
  imports: [PageWrapper, HasPermissionDirective, RouterLink, Table, TranslateModule],
})
export class Page {
  private store = inject(Store);
  router = inject(Router);

  page$: Observable<IPageModel> = inject(Store).select(PageState.page);
  setting$: Observable<IValues> = inject(Store).select(SettingState.setting) as Observable<IValues>;

  public url: string;

  public tableConfig: ITableConfig = {
    columns: [
      { title: 'title', dataField: 'title', sortable: true, sort_direction: 'desc' },
      {
        title: 'created_at',
        dataField: 'created_at',
        type: 'date',
        sortable: true,
        sort_direction: 'desc',
      },
      { title: 'status', dataField: 'status', type: 'switch' },
    ],
    rowActions: [
      { label: 'Edit', actionToPerform: 'edit', icon: 'ri-pencil-line', permission: 'page.edit' },
      {
        label: 'Delete',
        actionToPerform: 'delete',
        icon: 'ri-delete-bin-line',
        permission: 'page.destroy',
      },
      { label: 'View', actionToPerform: 'view', icon: 'ri-eye-line' },
    ],
    search: true,
    data: [] as IPage[],
    total: 0,
  };

  constructor() {
    this.setting$.subscribe(setting => {
      this.url = setting.general.site_url;
    });
  }

  ngOnInit(): void {
    this.page$.subscribe(page => {
      this.tableConfig.data = page ? page?.data : [];
      this.tableConfig.total = page ? page?.total : 0;
    });
  }

  onTableChange(data?: Params) {
    this.store.dispatch(new GetPagesAction(data));
  }

  onActionClicked(action: ITableClickedAction) {
    if (action.actionToPerform == 'edit') this.edit(action.data);
    else if (action.actionToPerform == 'status') this.status(action.data);
    else if (action.actionToPerform == 'delete') this.delete(action.data);
    else if (action.actionToPerform == 'deleteAll') this.deleteAll(action.data);
    else if (action.actionToPerform == 'view') this.view(action.data);
  }

  edit(data: IPage) {
    void this.router.navigateByUrl(
      `/page/${this.store.selectSnapshot(state => state.language.localLanguage.locale)}/edit/${data.id}`,
    );
  }

  status(data: IPage) {
    this.store.dispatch(new UpdatePageStatusAction(data.id, data.status));
  }

  delete(data: IPage) {
    this.store.dispatch(new DeletePageAction(data.id));
  }

  deleteAll(ids: number[]) {
    this.store.dispatch(new DeleteAllPageAction(ids));
  }

  view(data: IPage) {
    window.open(this.url + '/page/' + data.slug, '_blank');
  }
}
