import { AsyncPipe, CommonModule } from '@angular/common';
import { Component, inject, input } from '@angular/core';
import {
  FormBuilder,
  FormControl,
  FormGroup,
  FormsModule,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { ActivatedRoute, Router } from '@angular/router';

import { TranslatePipe } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Select2Data, Select2Module, Select2UpdateEvent } from 'ng-select2-component';
import { map, mergeMap, Observable, of, Subject, switchMap, takeUntil } from 'rxjs';

import { EditFormLanguageAction } from '../../../shared/action/language.action';
import {
  CreatePublicationAction,
  EditPublicationAction,
  UpdatePublicationAction,
} from '../../../shared/action/publication.action';
import { Button } from '../../../shared/components/ui/button/button';
import { FormFields } from '../../../shared/components/ui/form-fields/form-fields';
import { ImageUpload } from '../../../shared/components/ui/image-upload/image-upload';
import { LanguageRedirection } from '../../../shared/components/ui/language-redirection/language-redirection';
import * as data from '../../../shared/data/media-config';
import { IAttachment } from '../../../shared/interface/attachment.interface';
import { IPublication } from '../../../shared/interface/publication.interface';
import { IStores } from '../../../shared/interface/store.interface';
import { CountryState } from '../../../shared/state/country.state';
import { PublicationState } from '../../../shared/state/publication.state';
import { StateState } from '../../../shared/state/state.state';
import { StoreState } from '../../../shared/state/store.state';

@Component({
  selector: 'app-form-publication',
  imports: [
    CommonModule,
    FormFields,
    FormsModule,
    ReactiveFormsModule,
    TranslatePipe,
    Button,
    ImageUpload,
    TranslatePipe,
    Select2Module,
    AsyncPipe,
    LanguageRedirection,
  ],
  templateUrl: './form-publication.html',
  styleUrl: './form-publication.scss',
})
export class FormPublication {
  private store = inject(Store);
  private route = inject(ActivatedRoute);
  private router = inject(Router);
  private formBuilder = inject(FormBuilder);

  readonly type = input<string>();

  publication$: Observable<IPublication> = inject(Store).select(
    PublicationState.selectedPublication,
  ) as Observable<IPublication>;
  store$: Observable<IStores> = inject(Store).select(
    StoreState.selectedStore,
  ) as Observable<IStores>;
  countries$: Observable<Select2Data> = inject(Store).select(CountryState.countries);

  public form: FormGroup;
  public id: number;
  public lang: string;
  public mediaConfig: data.IMediaConfig = data.mediaConfig;
  public states$: Observable<Select2Data>;
  private destroy$ = new Subject<void>();

  constructor() {
    this.form = this.formBuilder.group({
      publisher_name: new FormControl('', [Validators.required]),
      slug: new FormControl(''),
      description: new FormControl('', [Validators.required]),
      publisher_logo_id: new FormControl('', [Validators.required]),
      publisher_cover_image_id: new FormControl(),
      country_id: new FormControl('', [Validators.required]),
      state_id: new FormControl('', [Validators.required]),
      city: new FormControl('', [Validators.required]),
      facebook: new FormControl(''),
      twitter: new FormControl(''),
      instagram: new FormControl(''),
      youtube: new FormControl(''),
      pinterest: new FormControl(''),
      status: new FormControl(1),
    });
  }

  ngOnInit() {
    this.route.params
      .pipe(
        switchMap(params => {
          if (!params['id']) return of();
          this.states$ = this.store.select(StateState.states).pipe(map(filterFn => filterFn(null)));
          if (params['lang']) this.lang = params['lang'];
          this.store.dispatch(new EditFormLanguageAction(params['lang']));
          return this.store
            .dispatch(new EditPublicationAction(params['id']))
            .pipe(mergeMap(() => this.store.select(PublicationState.selectedPublication)));
        }),
        takeUntil(this.destroy$),
      )
      .subscribe(publication => {
        this.id = publication?.id!;
        this.form.patchValue({
          publisher_name: publication?.publisher_name,
          slug: publication?.slug,
          publisher_logo_id: publication?.publisher_logo_id,
          publisher_cover_image_id: publication?.publisher_cover_image_id,
          country_id: publication?.country_id,
          state_id: publication?.state_id,
          city: publication?.city,
          facebook: publication?.facebook,
          twitter: publication?.twitter,
          instagram: publication?.instagram,
          youtube: publication?.youtube,
          pinterest: publication?.pinterest,
          status: publication?.status,
        });
      });
  }

  countryChange(data: Select2UpdateEvent) {
    if (data && data?.value) {
      this.states$ = this.store
        .select(StateState.states)
        .pipe(map(filterFn => filterFn(+data?.value)));

      if (data.component.focused) this.form.controls['state_id'].setValue('');
    } else {
      this.form.controls['state_id'].setValue('');
    }
  }

  selectPublisherLogo(data: IAttachment) {
    if (!Array.isArray(data)) {
      this.form.controls['publisher_logo_id'].setValue(data ? data.id : null);
    }
  }

  selectPublisherCoverImage(data: IAttachment) {
    if (!Array.isArray(data)) {
      this.form.controls['publisher_cover_image_id'].setValue(data ? data.id : null);
    }
  }

  submit() {
    this.form.markAllAsTouched();
    let action = new CreatePublicationAction(this.form.value);

    if (this.type() == 'edit' && this.id) {
      action = new UpdatePublicationAction(this.form.value, this.id);
    }

    if (this.form.valid) {
      this.store.dispatch(action).subscribe({
        complete: () => {
          void this.router.navigateByUrl('/publication');
          this.form.reset();
        },
      });
    }
  }

  ngOnDestroy() {
    this.destroy$.next();
    this.destroy$.complete();
  }
}
