import { Component, inject } from '@angular/core';
import { Router, RouterLink } from '@angular/router';

import { TranslatePipe } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Observable } from 'rxjs';

import {
  DeleteAllPublicationAction,
  DeletePublicationAction,
  GetPublicationsAction,
  UpdatePublicationStatusAction,
} from '../../shared/action/publication.action';
import { PageWrapper } from '../../shared/components/page-wrapper/page-wrapper';
import { Table } from '../../shared/components/ui/table/table';
import { HasPermissionDirective } from '../../shared/directive/has-permission.directive';
import { Params } from '../../shared/interface/core.interface';
import { IPublication, IPublicationModel } from '../../shared/interface/publication.interface';
import { ITableClickedAction, ITableConfig } from '../../shared/interface/table.interface';
import { PublicationState } from '../../shared/state/publication.state';

@Component({
  selector: 'app-publication',
  imports: [PageWrapper, Table, TranslatePipe, RouterLink, HasPermissionDirective],
  templateUrl: './publication.html',
  styleUrl: './publication.scss',
})
export class Publication {
  private store = inject(Store);
  router = inject(Router);

  publication$: Observable<IPublicationModel> = inject(Store).select(PublicationState.publication);

  public tableConfig: ITableConfig = {
    columns: [
      { title: 'Name', dataField: 'publisher_name', sortable: true, sort_direction: 'desc' },
      {
        title: 'created_at',
        dataField: 'created_at',
        type: 'date',
        sortable: true,
        sort_direction: 'desc',
      },
      { title: 'status', dataField: 'status', type: 'switch' },
    ],
    rowActions: [
      { label: 'Edit', actionToPerform: 'edit', icon: 'ri-pencil-line', permission: 'brand.edit' },
      {
        label: 'Delete',
        actionToPerform: 'delete',
        icon: 'ri-delete-bin-line',
        permission: 'brand.destroy',
      },
    ],
    search: true,
    data: [] as IPublication[],
    total: 0,
  };

  ngOnInit() {
    this.publication$.subscribe(publication => {
      this.tableConfig.data = publication ? publication?.data : [];
      this.tableConfig.total = publication ? publication?.total : 0;
    });
  }

  onTableChange(data?: Params) {
    this.store.dispatch(new GetPublicationsAction(data));
  }

  onActionClicked(action: ITableClickedAction) {
    if (action.actionToPerform == 'edit') this.edit(action.data);
    else if (action.actionToPerform == 'status') this.status(action.data);
    else if (action.actionToPerform == 'delete') this.delete(action.data);
    else if (action.actionToPerform == 'deleteAll') this.deleteAll(action.data);
  }

  edit(data: IPublication) {
    void this.router.navigateByUrl(
      `/publication/${this.store.selectSnapshot(state => state.language.localLanguage.locale)}/edit/${data.id}`,
    );
  }

  status(data: IPublication) {
    this.store.dispatch(new UpdatePublicationStatusAction(data.id, data.status));
  }

  delete(data: IPublication) {
    this.store.dispatch(new DeletePublicationAction(data.id));
  }

  deleteAll(ids: number[]) {
    this.store.dispatch(new DeleteAllPublicationAction(ids));
  }
}
