import { AsyncPipe, CommonModule } from '@angular/common';
import { Component, inject, Renderer2, DOCUMENT } from '@angular/core';
import { FormBuilder, FormGroup } from '@angular/forms';
import { Params, Router } from '@angular/router';

import {
  NgbDate,
  NgbDateParserFormatter,
  NgbDatepickerModule,
  NgbInputDatepicker,
} from '@ng-bootstrap/ng-bootstrap';
import { TranslatePipe } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Select2Data, Select2Module, Select2UpdateEvent } from 'ng-select2-component';
import { Observable } from 'rxjs';

import {
  GetReportAction,
  GetReportsFieldsAction,
  GetReportsTypesAction,
} from '../../shared/action/reports.action';
import { PageWrapper } from '../../shared/components/page-wrapper/page-wrapper';
import { Button } from '../../shared/components/ui/button/button';
import { FormFields } from '../../shared/components/ui/form-fields/form-fields';
import { Table } from '../../shared/components/ui/table/table';
import { ICoupon } from '../../shared/interface/coupon.interface';
import { IReportsCouponModel } from '../../shared/interface/reports.interface';
import { ITableConfig } from '../../shared/interface/table.interface';
import { ReportsState } from '../../shared/state/reports.state';

@Component({
  selector: 'app-reports',
  imports: [
    CommonModule,
    PageWrapper,
    Select2Module,
    NgbDatepickerModule,
    NgbInputDatepicker,
    FormFields,
    TranslatePipe,
    Table,
    AsyncPipe,
    Button,
  ],
  templateUrl: './reports.html',
  styleUrl: './reports.scss',
})
export class Reports {
  private store = inject(Store);
  private document = inject<Document>(DOCUMENT);
  formatter = inject(NgbDateParserFormatter);
  private renderer = inject(Renderer2);
  private fb = inject(FormBuilder);
  router = inject(Router);

  public reports$: Observable<IReportsCouponModel> = inject(Store).select(ReportsState.reports);
  public reportsType$: Observable<Select2Data> = inject(Store).select(ReportsState.reportsTypes);
  public reportsFields$: Observable<any> = inject(Store).select(ReportsState.reportsFields);
  public hoveredDate: NgbDate | null = null;
  public fromDate: NgbDate | null;
  public toDate: NgbDate | null;
  public report: string;
  public filter: Params;
  public reportsFields: any;

  public tableConfig: ITableConfig = {
    columns: [],
    data: [] as ICoupon[],
    total: 0,
    search: false,
  };

  public form: FormGroup;

  constructor() {
    this.form = this.fb.group({});
    this.store.dispatch(new GetReportsTypesAction());
  }

  ngOnInit() {
    this.reportsFields$.subscribe(res => {
      this.reportsFields = res;

      res.fields?.length &&
        res.fields.forEach((field: any) => {
          if (field.type === 'text') {
            this.form.addControl(field.name, this.fb.control(''));
          } else if (field.type === 'select') {
            this.form.addControl(field.name, this.fb.control(''));
          } else if (field.type === 'date') {
            this.form.addControl(field.name, this.fb.control(''));
          }
        });
    });

    this.reports$.subscribe(report => {
      // Validate report data
      if (!report?.data || !Array.isArray(report.data)) {
        console.error('Invalid report data:', report);
        this.tableConfig.data = [];
        this.tableConfig.total = 0;
        return;
      }

      // Assign report data
      const reports = report.data;

      // Update table configuration
      this.tableConfig.data = reports.map((item: any) => {
        // Map nested properties to a flat structure
        const mappedItem: any = {};
        this.reportsFields?.show_fields_table.length &&
          this.reportsFields?.show_fields_table?.forEach((key: any) => {
            if (key.includes('.')) {
              const value = this.getNestedValue(item, key.split('.'));
              const lastKey = key.split('.').pop(); // Get the last part of the key
              if (lastKey) mappedItem[lastKey] = value;
            } else {
              mappedItem[key] = item[key];
            }
          });
        return mappedItem;
      });

      this.tableConfig.total = report.total || 0;

      // Configure table columns
      this.tableConfig.columns = this.reportsFields?.show_fields_table?.map((key: any) => ({
        title: key.split('.').pop() || key, // Use the last part of the key for title
        dataField: key.split('.').pop() || key, // Use the last part of the key for dataField
        sortable: true,
        sort_direction: 'desc',
      }));
    });
  }

  getNestedValue(obj: any, path: string[]): any {
    return path.reduce((acc, key) => (acc && acc[key] !== undefined ? acc[key] : undefined), obj);
  }

  onTableChange(data?: Params) {
    this.filter = { ...data!, ...this.filter };
    if (this.report) this.store.dispatch(new GetReportAction(this.report, this.filter)).subscribe();
  }

  onDateSelection(date: NgbDate) {
    if (!this.fromDate && !this.toDate) {
      this.fromDate = date;
    } else if (this.fromDate && !this.toDate && date && date.after(this.fromDate)) {
      this.toDate = date;
    } else {
      this.toDate = null;
      this.fromDate = date;
    }

    if (this.fromDate)
      this.form.controls['start_date'].setValue(
        `${this.fromDate.year}-${this.fromDate.month}-${this.fromDate.day}`,
      );
    if (this.toDate)
      this.form.controls['end_date'].setValue(
        `${this.toDate?.year}-${this.toDate?.month}-${this.toDate?.day}`,
      );
  }

  selectType(data: Select2UpdateEvent) {
    if (data && data?.value) {
      this.form.reset();
      this.report = String(data.value);
      this.renderer.addClass(this.document.body, 'loader-none');
      this.store.dispatch(new GetReportsFieldsAction(String(data?.value))).subscribe({
        next: () => {
          this.store.dispatch(new GetReportAction(this.report, this.filter));
        },
      });
    }
  }

  submit() {
    let filter = this.form.value;
    this.filter = { ...this.filter, ...filter };
    this.store.dispatch(new GetReportAction(this.report, this.filter)).subscribe();
  }

  ngOnDestroy() {
    this.renderer.removeClass(this.document.body, 'loader-none');
  }
}
