import { Component, inject, input } from '@angular/core';
import {
  FormBuilder,
  FormControl,
  FormGroup,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { ActivatedRoute, Router } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Subject, of } from 'rxjs';
import { mergeMap, switchMap, takeUntil } from 'rxjs/operators';

import { EditFormLanguageAction } from '../../../shared/action/language.action';
import { CreateTagAction, EditTagAction, UpdateTagAction } from '../../../shared/action/tag.action';
import { Button } from '../../../shared/components/ui/button/button';
import { FormFields } from '../../../shared/components/ui/form-fields/form-fields';
import { LanguageRedirection } from '../../../shared/components/ui/language-redirection/language-redirection';
import { ITag } from '../../../shared/interface/tag.interface';
import { TagState } from '../../../shared/state/tag.state';

@Component({
  selector: 'app-form-tag',
  templateUrl: './form-tag.html',
  styleUrls: ['./form-tag.scss'],
  imports: [LanguageRedirection, ReactiveFormsModule, FormFields, Button, TranslateModule],
})
export class FormTag {
  private store = inject(Store);
  private router = inject(Router);
  private route = inject(ActivatedRoute);
  private formBuilder = inject(FormBuilder);

  readonly type = input<string>();
  readonly tagType = input<string | null>('product');

  public form: FormGroup;
  public tag: ITag | null;
  public lang: string;

  private destroy$ = new Subject<void>();

  constructor() {
    this.form = this.formBuilder.group({
      name: new FormControl('', [Validators.required]),
      slug: new FormControl(''),
      description: new FormControl(),
      type: new FormControl(this.tagType(), []),
      status: new FormControl(true),
    });
  }

  ngOnChanges() {
    this.form.controls['type'].setValue(this.tagType());
  }

  ngOnInit() {
    this.route.params
      .pipe(
        switchMap(params => {
          if (!params['id']) return of();

          if (params['lang']) this.lang = params['lang'];
          this.store.dispatch(new EditFormLanguageAction(params['lang']));

          return this.store
            .dispatch(new EditTagAction(params['id']))
            .pipe(mergeMap(() => this.store.select(TagState.selectedTag)));
        }),
        takeUntil(this.destroy$),
      )
      .subscribe(tag => {
        this.tag = tag;
        this.form.patchValue({
          name: this.tag?.name,
          slug: this.tag?.slug,
          description: this.tag?.description,
          status: this.tag?.status,
        });
      });
  }

  submit() {
    this.form.markAllAsTouched();
    let action = new CreateTagAction(this.form.value);

    if (this.type() == 'edit' && this.tag?.id) {
      action = new UpdateTagAction(this.form.value, this.tag.id);
    }

    if (this.form.valid) {
      this.store.dispatch(action).subscribe({
        complete: () => {
          if (this.tagType() == 'post') void this.router.navigateByUrl('/blog/tag');
          else void this.router.navigateByUrl('/tag');
        },
      });
    }
  }
}
