import { AsyncPipe } from '@angular/common';
import { Component, PLATFORM_ID, inject, input } from '@angular/core';
import {
  FormBuilder,
  FormControl,
  FormGroup,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { ActivatedRoute, Router } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Select2Data, Select2Module } from 'ng-select2-component';
import { Observable, Subject, forkJoin, of } from 'rxjs';
import { mergeMap, switchMap, takeUntil } from 'rxjs/operators';

import { GetRolesAction } from '../../../shared/action/role.action';
import {
  CreateUserAction,
  EditUserAction,
  UpdateUserAction,
} from '../../../shared/action/user.action';
import { Button } from '../../../shared/components/ui/button/button';
import { FormFields } from '../../../shared/components/ui/form-fields/form-fields';
import * as data from '../../../shared/data/country-code';
import { RoleState } from '../../../shared/state/role.state';
import { UserState } from '../../../shared/state/user.state';
import { CustomValidators } from '../../../shared/validator/password-match';

@Component({
  selector: 'app-form-user',
  templateUrl: './form-user.html',
  styleUrls: ['./form-user.scss'],
  imports: [
    ReactiveFormsModule,
    FormFields,
    Select2Module,
    Button,
    AsyncPipe,
    TranslateModule,
    Select2Module,
  ],
})
export class FormUser {
  private store = inject(Store);
  private route = inject(ActivatedRoute);
  private router = inject(Router);
  private formBuilder = inject(FormBuilder);
  private platformId = inject<Object>(PLATFORM_ID);

  readonly type = input<string>();

  role$: Observable<Select2Data> = inject(Store).select(RoleState.roles);

  public form: FormGroup;
  public id: number;
  public codes = data.countryCodes;

  private destroy$ = new Subject<void>();

  constructor() {
    this.form = this.formBuilder.group(
      {
        name: new FormControl('', [Validators.required]),
        email: new FormControl('', [Validators.required, Validators.email]),
        phone: new FormControl('', [Validators.required, Validators.pattern(/^[0-9]*$/)]),
        country_code: new FormControl('91', [Validators.required]),
        role_id: new FormControl('', [Validators.required]),
        password: new FormControl('', [Validators.required]),
        password_confirmation: new FormControl('', [Validators.required]),
        status: new FormControl(1),
      },
      {
        validator: CustomValidators.MatchValidator('password', 'password_confirmation'),
      },
    );
  }

  get passwordMatchError() {
    return this.form.getError('mismatch') && this.form.get('password_confirmation')?.touched;
  }

  ngOnInit() {
    const roles$ = this.store.dispatch(new GetRolesAction());
    this.route.params
      .pipe(
        switchMap(params => {
          if (!params['id']) return of();
          return this.store
            .dispatch(new EditUserAction(params['id']))
            .pipe(mergeMap(() => this.store.select(UserState.selectedUser)));
        }),
        takeUntil(this.destroy$),
      )
      .subscribe(user => {
        this.id = user?.id!;
        forkJoin([roles$]).subscribe({
          complete: () => {
            this.form.patchValue({
              name: user?.name,
              email: user?.email,
              phone: user?.phone,
              country_code: user?.country_code,
              role_id: user?.role ? user?.role.id : null,
              status: user?.status,
            });
          },
        });
      });
  }

  submit() {
    this.form.markAllAsTouched();
    let action = new CreateUserAction(this.form.value);

    if (this.type() == 'edit' && this.id) {
      this.form.removeControl('password');
      this.form.removeControl('password_confirmation');
      action = new UpdateUserAction(this.form.value, this.id);
    }

    if (this.form.valid) {
      this.store.dispatch(action).subscribe({
        complete: () => {
          void this.router.navigateByUrl('/user');
        },
      });
    }
  }

  ngOnDestroy() {
    this.destroy$.next();
    this.destroy$.complete();
  }
}
