import { CommonModule } from '@angular/common';
import { Component, NgZone, inject, input } from '@angular/core';
import {
  FormBuilder,
  FormControl,
  FormGroup,
  FormsModule,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { GoogleMap } from '@angular/google-maps';
import { ActivatedRoute, Router } from '@angular/router';

import { TranslatePipe } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { mergeMap, of, Subject, switchMap, takeUntil } from 'rxjs';

import {
  CreateZoneAction,
  EditZoneAction,
  UpdateZoneAction,
} from '../../../shared/action/zone.action';
import { Button } from '../../../shared/components/ui/button/button';
import { FormFields } from '../../../shared/components/ui/form-fields/form-fields';
import { IZone } from '../../../shared/interface/zone.interface';
import { ZoneState } from '../../../shared/state/zone.state';

@Component({
  selector: 'app-form-zone',
  imports: [
    CommonModule,
    GoogleMap,
    FormsModule,
    ReactiveFormsModule,
    TranslatePipe,
    Button,
    FormFields,
  ],
  templateUrl: './form-zone.html',
  styleUrl: './form-zone.scss',
})
export class FormZone {
  private store = inject(Store);
  private router = inject(Router);
  private route = inject(ActivatedRoute);
  private ngZone = inject(NgZone);
  private formBuilder = inject(FormBuilder);

  readonly type = input<string>();

  map!: google.maps.Map;
  markers: google.maps.Marker[] = [];
  polygon!: google.maps.Polygon;
  coordinates: { lat: number; lng: number }[] = [];
  lastActiveMarker: google.maps.Marker | null = null;

  // Undo/Redo stacks
  public undoStack: { lat: number; lng: number }[] = [];
  public redoStack: { lat: number; lng: number }[] = [];

  public form: FormGroup;
  public zone: IZone | null;
  public lang: string;

  private destroy$ = new Subject<void>();

  constructor() {
    this.form = this.formBuilder.group({
      name: new FormControl('', [Validators.required]),
      place_points: new FormControl([], [Validators.required]),
      status: new FormControl(true),
    });
  }

  ngOnInit() {
    this.initializeMap();

    this.route.params
      .pipe(
        switchMap(params => {
          if (!params['id']) return of();

          return this.store
            .dispatch(new EditZoneAction(params['id']))
            .pipe(mergeMap(() => this.store.select(ZoneState.selectedZone)));
        }),
        takeUntil(this.destroy$),
      )
      .subscribe(zone => {
        this.zone = zone;

        this.form.patchValue({
          name: this.zone?.name,
          place_points: this.zone?.locations, // Adjust if necessary
          status: this.zone?.status,
        });

        // Initialize markers and polygon on the map
        if (this.zone?.locations) {
          this.zone.locations.forEach(location => {
            this.addMarker(new google.maps.LatLng(location.lat, location.lng));
          });

          this.setMapCenter(this.coordinates); // Center map to polygon
        }
      });
  }

  initializeMap() {
    const mapOptions: google.maps.MapOptions = {
      center: { lat: 37.7749, lng: -122.4194 }, // Default center (e.g., San Francisco)
      zoom: 12,
    };

    this.map = new google.maps.Map(document.getElementById('map') as HTMLElement, mapOptions);

    // Add click listener to the map
    this.map.addListener('click', (event: google.maps.MapMouseEvent) => {
      if (event.latLng) {
        this.addMarker(event.latLng);
      }
    });
  }

  addMarker(latLng: google.maps.LatLng) {
    // Set the paths for your custom PNG icons
    const inactiveIcon = 'assets/images/green-map.svg'; // Path to black circle PNG
    const activeIcon = 'assets/images/red-map.svg'; // Path to red circle PNG

    // Create a new marker with the inactive icon by default
    const marker = new google.maps.Marker({
      position: latLng,
      map: this.map,
      icon: inactiveIcon, // Default icon
    });

    // Update the last active marker's appearance
    if (this.lastActiveMarker) {
      this.lastActiveMarker.setIcon(inactiveIcon); // Reset the previous marker to default
    }
    marker.setIcon(activeIcon); // Highlight the newly added marker
    this.lastActiveMarker = marker;

    // Store the marker and coordinates
    this.ngZone.run(() => {
      this.markers.push(marker);
      this.coordinates.push({ lat: latLng.lat(), lng: latLng.lng() });
      this.undoStack.push({ lat: latLng.lat(), lng: latLng.lng() });

      // Clear redo stack since a new action invalidates the redo history
      this.redoStack = [];
    });

    // Check if enough points are available to draw a polygon
    if (this.coordinates.length >= 3) {
      this.drawPolygon();
    }
  }

  drawPolygon() {
    // Remove existing polygon, if any
    if (this.polygon) {
      this.polygon.setMap(null);
    }

    // Create a polygon from the coordinates
    this.polygon = new google.maps.Polygon({
      paths: this.coordinates,
      strokeColor: '#000000',
      strokeOpacity: 0.8,
      strokeWeight: 2,
      fillColor: '#626262',
      fillOpacity: 0.35,
    });

    // Add the polygon to the map
    this.polygon.setMap(this.map);
  }
  undo() {
    if (this.undoStack.length === 0) return;

    const removedPoint = this.undoStack.pop();
    if (removedPoint) {
      this.redoStack.push(removedPoint); // Push the removed point to redoStack
      this.coordinates.pop(); // Remove from coordinates array

      // Remove last marker
      const marker = this.markers.pop();
      if (marker) marker.setMap(null);
    }

    // Update active marker
    if (this.markers.length > 0) {
      this.lastActiveMarker = this.markers[this.markers.length - 1];
      this.lastActiveMarker.setIcon('assets/images/red-map.svg'); // Set active
    } else {
      this.lastActiveMarker = null;
    }

    this.drawPolygon();
  }

  redo() {
    if (this.redoStack.length === 0) return;

    const restoredPoint = this.redoStack.pop();
    if (restoredPoint) {
      this.undoStack.push(restoredPoint); // Push back to undoStack
      this.coordinates.push(restoredPoint);

      const marker = new google.maps.Marker({
        position: restoredPoint,
        map: this.map,
        icon: 'assets/images/details_open.png',
      });

      // Update active marker
      if (this.lastActiveMarker) {
        this.lastActiveMarker.setIcon('assets/images/details_open.png');
      }
      marker.setIcon('assets/images/red-map.svg');
      this.lastActiveMarker = marker;

      this.markers.push(marker);
    }

    this.drawPolygon();
  }

  // Dynamically center the map based on coordinates
  setMapCenter(coordinates: { lat: number; lng: number }[]) {
    if (coordinates.length === 0) return;

    const bounds = new google.maps.LatLngBounds();
    coordinates.forEach(coord => {
      bounds.extend(new google.maps.LatLng(coord.lat, coord.lng));
    });

    this.map.fitBounds(bounds); // Adjust the map to fit the polygon
  }

  submit() {
    this.form.get('place_points')?.setValue(this.coordinates);
    this.form.markAllAsTouched();
    let action = new CreateZoneAction(this.form.value);
    if (this.type() == 'edit' && this.zone?.id) {
      action = new UpdateZoneAction(this.form.value, this.zone.id);
    }

    if (this.form.valid) {
      this.store.dispatch(action).subscribe({
        complete: () => {
          void this.router.navigateByUrl('/zone');
        },
      });
    }
  }
}
