import { CommonModule } from '@angular/common';
import { Component, inject } from '@angular/core';
import { Params, Router, RouterLink } from '@angular/router';

import { TranslatePipe } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Observable } from 'rxjs';

import {
  DeleteAllZoneAction,
  DeleteZoneAction,
  GetZonesAction,
  UpdateZoneStatusAction,
} from '../../shared/action/zone.action';
import { PageWrapper } from '../../shared/components/page-wrapper/page-wrapper';
import { Table } from '../../shared/components/ui/table/table';
import { HasPermissionDirective } from '../../shared/directive/has-permission.directive';
import { ITableClickedAction, ITableConfig } from '../../shared/interface/table.interface';
import { IZoneModel } from '../../shared/interface/zone.interface';
import { ZoneState } from '../../shared/state/zone.state';

@Component({
  selector: 'app-zone',
  imports: [CommonModule, PageWrapper, Table, TranslatePipe, RouterLink, HasPermissionDirective],
  templateUrl: './zone.html',
  styleUrl: './zone.scss',
})
export class Zone {
  private store = inject(Store);
  router = inject(Router);

  zone$: Observable<IZoneModel> = inject(Store).select(ZoneState.zone);

  public tableConfig: ITableConfig = {
    columns: [
      { title: 'name', dataField: 'name', sortable: true, sort_direction: 'desc' },
      {
        title: 'created_at',
        dataField: 'created_at',
        type: 'date',
        sortable: true,
        sort_direction: 'desc',
      },
      { title: 'status', dataField: 'status', type: 'switch' },
    ],
    rowActions: [
      { label: 'Edit', actionToPerform: 'edit', icon: 'ri-pencil-line', permission: 'zone.edit' },
      {
        label: 'Delete',
        actionToPerform: 'delete',
        icon: 'ri-delete-bin-line',
        permission: 'zone.destroy',
      },
    ],
    data: [] as any[],
    search: true,
    total: 0,
  };

  ngOnInit() {
    this.zone$.subscribe(zone => {
      this.tableConfig.data = zone ? zone?.data : [];
      this.tableConfig.total = zone ? zone?.total : 0;
    });
  }

  onTableChange(data?: Params) {
    this.store.dispatch(new GetZonesAction(data));
  }

  onActionClicked(action: ITableClickedAction) {
    if (action.actionToPerform == 'edit') this.edit(action.data);
    else if (action.actionToPerform == 'status') this.status(action.data);
    else if (action.actionToPerform == 'delete') this.delete(action.data);
    else if (action.actionToPerform == 'deleteAll') this.deleteAll(action.data);
  }

  edit(data: any) {
    void this.router.navigateByUrl(
      `/zone/${this.store.selectSnapshot(state => state.language.localLanguage.locale)}/edit/${data.id}`,
    );
  }

  status(data: any) {
    this.store.dispatch(new UpdateZoneStatusAction(data.id, data.status));
  }

  delete(data: any) {
    this.store.dispatch(new DeleteZoneAction(data.id));
  }

  deleteAll(ids: number[]) {
    this.store.dispatch(new DeleteAllZoneAction(ids));
  }
}
