import { Injectable, PLATFORM_ID, inject } from '@angular/core';
import { UrlTree, Router } from '@angular/router';

import { Store } from '@ngxs/store';
import { Observable, catchError, delay, map, of, switchMap, tap } from 'rxjs';

import { GetBadgesAction } from '../../shared/action/sidebar.action';
import { GetUserDetailsAction } from './../../shared/action/account.action';
import { GetNotificationAction } from './../../shared/action/notification.action';
import { NavService } from './../../shared/services/nav.service';

@Injectable({
  providedIn: 'root',
})
export class AuthGuard {
  private store = inject(Store);
  private router = inject(Router);
  private navService = inject(NavService);
  private platformId = inject<Object>(PLATFORM_ID);

  // Inside your AuthGuard
  canActivate(): Observable<boolean | UrlTree> | boolean | UrlTree {
    if (!this.store.selectSnapshot(state => state.auth && state.auth.access_token)) {
      void this.router.navigate(['/auth/login']);
      return false;
    }

    this.navService.sidebarLoading = true;

    this.store.dispatch(new GetBadgesAction());
    this.store.dispatch(new GetNotificationAction());
    this.store.dispatch(new GetUserDetailsAction()).subscribe({
      complete: () => {
        this.navService.sidebarLoading = false;
      },
    });

    return true;
  }

  canActivateChild(): Observable<boolean> {
    return this.checkAuthStatus().pipe(
      switchMap(isAuthenticated => {
        if (isAuthenticated) {
          return of(true).pipe(
            delay(1000),
            tap(() => this.router.navigate(['/dashboard'])),
          );
        }

        return of(true);
      }),
    );
  }

  private checkAuthStatus(): Observable<boolean> {
    return this.store
      .select(state => state.auth && state.auth.access_token)
      .pipe(
        map(access_token => !!access_token),
        catchError(() => of(false)),
      );
  }
}
