import {
  HttpRequest,
  HttpHandler,
  HttpInterceptor,
  HttpErrorResponse,
  HttpEvent,
} from '@angular/common/http';
import { Injectable, inject } from '@angular/core';
import { Router } from '@angular/router';

import { Store } from '@ngxs/store';
import { catchError, Observable, throwError } from 'rxjs';

import { AuthClearAction } from '../../shared/action/auth.action';
import { NotificationService } from '../../shared/services/notification.service';

@Injectable()
export class AuthInterceptor implements HttpInterceptor {
  private router = inject(Router);
  private store = inject(Store);
  private notificationService = inject(NotificationService);

  intercept<T>(req: HttpRequest<T>, next: HttpHandler): Observable<HttpEvent<T>> {
    const token = this.store.selectSnapshot(
      state => state && state.auth && state.auth.access_token,
    );
    const lang = this.store.selectSnapshot(state => state.language.localLanguage);
    const langFrom = this.store.selectSnapshot(state => state.language.formLanguage);
    console.warn(
      '🚀 ~ AuthInterceptor ~ intercept ~ langFrom:',
      req.url,
      langFrom && !req.url.includes('translation/admin'),
      langFrom,
      lang?.locale,
    );

    // Prepare headers
    const headers: Record<string, string> = {
      'Accept-Lang':
        langFrom && !req.url.includes('translation/admin') ? langFrom : lang?.locale || 'en',
    };

    if (token) {
      headers['Authorization'] = `Bearer ${token}`;
    }

    // Clone the request with the headers
    const clonedRequest = req.clone({ setHeaders: headers });

    // Handle request and catch errors
    return next.handle(clonedRequest).pipe(
      catchError((error: HttpErrorResponse) => {
        if (error.status === 401) {
          this.notificationService.notification = false;
          this.store.dispatch(new AuthClearAction());
          void this.router.navigate(['/auth/login']);
        }
        return throwError(() => error);
      }),
    );
  }
}
