import { CommonModule } from '@angular/common';
import { Component, DOCUMENT, inject } from '@angular/core';
import { RouterModule } from '@angular/router';

import { TranslateModule } from '@ngx-translate/core';
import { Store } from '@ngxs/store';
import { Observable } from 'rxjs';

import { HasPermissionDirective } from '../../directive/has-permission.directive';
import { IAccountUser } from '../../interface/account.interface';
import { INotification } from '../../interface/notification.interface';
import { IValues } from '../../interface/setting.interface';
import { NavService } from '../../services/nav.service';
import { AccountState } from '../../state/account.state';
import { NotificationState } from '../../state/notification.state';
import { SettingState } from '../../state/setting.state';
import { Languages } from './widgets/languages/languages';
import { Mode } from './widgets/mode/mode';
import { Notification } from './widgets/notification/notification';
import { Profile } from './widgets/profile/profile';
import { QuickView } from './widgets/quick-view/quick-view';
import { Search } from './widgets/search/search';

interface ILanguage {
  language: string;
  code: string;
  icon: string;
}

@Component({
  selector: 'app-header',
  imports: [
    CommonModule,
    Profile,
    Mode,
    Notification,
    Languages,
    QuickView,
    TranslateModule,
    Search,
    RouterModule,
    HasPermissionDirective,
  ],
  templateUrl: './header.html',
  styleUrls: ['./header.scss'],
})
export class Header {
  navServices = inject(NavService);
  private document = inject(DOCUMENT);

  user$: Observable<IAccountUser> = inject(Store).select(AccountState.user);
  notification$: Observable<INotification[]> = inject(Store).select(NotificationState.notification);
  setting$: Observable<IValues> = inject(Store).select(SettingState.setting) as Observable<IValues>;

  public unreadNotificationCount: number;

  public active: boolean = false;
  public profileOpen: boolean = false;
  public open: boolean = false;

  public languages: ILanguage[] = [
    {
      language: 'English',
      code: 'en',
      icon: 'us',
    },
    {
      language: 'Français',
      code: 'fr',
      icon: 'fr',
    },
  ];

  public selectedLanguage: ILanguage = {
    language: 'English',
    code: 'en',
    icon: 'us',
  };
  public elem: HTMLElement;
  public url: string;

  constructor() {
    const document = this.document;

    this.notification$.subscribe(notification => {
      this.unreadNotificationCount = notification?.filter(item => !item.read_at)?.length;
    });
    this.setting$.subscribe(setting => {
      this.url = setting?.general?.site_url!;
      document.body.classList.add(setting?.general?.mode!);
    });

    this.elem = document.documentElement;
  }

  sidebarToggle() {
    this.navServices.collapseSidebar = !this.navServices.collapseSidebar;
  }

  clickHeaderOnMobile() {
    this.navServices.search = true;
  }

  toggleFullScreen() {
    this.navServices.fullScreen = !this.navServices.fullScreen;
    if (this.navServices.fullScreen) {
      if (this.elem.requestFullscreen) {
        void this.elem.requestFullscreen();
      } else if ('mozRequestFullScreen' in this.elem) {
        /* Firefox */
        void (
          this.elem as HTMLElement & {
            mozRequestFullScreen: () => Promise<void>;
          }
        ).mozRequestFullScreen();
      } else if ('webkitRequestFullscreen' in this.elem) {
        /* Chrome, Safari and Opera */
        void (
          this.elem as HTMLElement & {
            webkitRequestFullscreen: () => Promise<void>;
          }
        ).webkitRequestFullscreen();
      } else if ('msRequestFullscreen' in this.elem) {
        /* IE/Edge */
        void (
          this.elem as HTMLElement & {
            msRequestFullscreen: () => Promise<void>;
          }
        ).msRequestFullscreen();
      }
    } else {
      if (!this.document.exitFullscreen) {
        void this.document.exitFullscreen();
      } else if ('mozCancelFullScreen' in this.document) {
        /* Firefox */
        void (
          this.document as Document & {
            mozCancelFullScreen: () => Promise<void>;
          }
        ).mozCancelFullScreen();
      } else if ('webkitExitFullscreen' in this.document) {
        /* Chrome, Safari and Opera */
        void (
          this.document as Document & {
            webkitExitFullscreen: () => Promise<void>;
          }
        ).webkitExitFullscreen();
      } else if ('msExitFullscreen' in this.document) {
        /* IE/Edge */
        void (
          this.document as Document & { msExitFullscreen: () => Promise<void> }
        ).msExitFullscreen();
      }
    }
  }
}
