import { HttpClient } from '@angular/common/http';
import { Injectable, inject } from '@angular/core';

import { Observable } from 'rxjs';

import { environment } from '../../../environments/environment';
import { Params } from '../interface/core.interface';
import { IProduct, IProductModel } from '../interface/product.interface';

@Injectable({
  providedIn: 'root',
})
export class ProductService {
  private http = inject(HttpClient);

  getProducts(payload?: Params): Observable<IProductModel> {
    return this.http.get<IProductModel>(`${environment.URL}/product`, { params: payload });
  }

  createProduct(payload: IProduct): Observable<IProduct> {
    return this.http.post<IProduct>(`${environment.URL}/product`, payload);
  }

  editProduct(id: number): Observable<IProduct> {
    return this.http.get<IProduct>(`${environment.URL}/product/${id}`);
  }

  updateProduct(id: number, payload: IProduct): Observable<IProduct> {
    return this.http.put<IProduct>(`${environment.URL}/product/${id}`, payload);
  }

  updateProductStatus(id: number, status: boolean): Observable<IProduct> {
    return this.http.put<IProduct>(`${environment.URL}/product/${id}/${status}`, {});
  }

  approveProductStatus(id: number, status: boolean): Observable<IProduct> {
    return this.http.put<IProduct>(`${environment.URL}/product/approve/${id}/${status}`, {});
  }

  deleteProduct(id: number): Observable<number> {
    return this.http.delete<number>(`${environment.URL}/product/${id}`);
  }

  deleteAllProduct(ids: number[]): Observable<number> {
    return this.http.post<number>(`${environment.URL}/product/deleteAll`, { ids: ids });
  }

  replicateProduct(ids: number[]): Observable<number> {
    return this.http.post<number>(`${environment.URL}/product/replicate`, { ids: ids });
  }

  importProduct(payload: File[]): Observable<IProduct[]> {
    let form = new FormData();
    if (Array.isArray(payload) && payload.length) {
      payload.forEach((element: File, _index: number) => {
        form.append(`products`, element, element.name);
      });
    }
    return this.http.post<IProduct[]>(`${environment.URL}/product/csv/import`, form);
  }

  exportProduct(payload?: Params): Observable<Blob> {
    return this.http.post<Blob>(`${environment.URL}/product/csv/export`, payload, {
      responseType: 'blob' as 'json',
    });
  }

  download(payload: {
    product_id: number;
    variation_id?: number | null;
  }): Observable<{ download_link: string }> {
    return this.http.post<{ download_link: string }>(
      `${environment.URL}/download/admin/zip/link`,
      payload,
    );
  }
}
