import { Injectable, inject } from '@angular/core';
import { Router } from '@angular/router';

import { Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetUserDetailsAction,
  UpdateUserProfileAction,
  UpdateUserPasswordAction,
  AccountClearAction,
  UpdateStoreDetailsAction,
} from '../action/account.action';
import { IPermission } from '../interface/role.interface';
import { AccountService } from '../services/account.service';
import { NotificationService } from '../services/notification.service';
import { IAccountUser } from './../interface/account.interface';

export class AccountStateModel {
  user: IAccountUser | null | any;
  permissions: IPermission[];
  roleName: string | null;
}

@State<AccountStateModel>({
  name: 'account',
  defaults: {
    user: null,
    permissions: [],
    roleName: null,
  },
})
@Injectable()
export class AccountState {
  private accountService = inject(AccountService);
  private notificationService = inject(NotificationService);
  router = inject(Router);

  @Selector()
  static user(state: AccountStateModel) {
    return state.user;
  }

  @Selector()
  static permissions(state: AccountStateModel) {
    return state.permissions;
  }

  @Selector()
  static getRoleName(state: AccountStateModel) {
    return state.roleName;
  }

  @Action(GetUserDetailsAction)
  getUserDetails(ctx: StateContext<AccountStateModel>) {
    return this.accountService.getUserDetails().pipe(
      tap({
        next: result => {
          ctx.patchState({
            user: result,
            permissions: result.permission,
            roleName: result.role.name,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateUserProfileAction)
  updateProfile(ctx: StateContext<AccountStateModel>, { payload }: UpdateUserProfileAction) {
    return this.accountService.updateProfile(payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            user: result,
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Profile Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateUserPasswordAction)
  updatePassword(ctx: StateContext<AccountStateModel>, { payload }: UpdateUserPasswordAction) {
    return this.accountService.updatePassword(payload).pipe(
      tap({
        complete: () => {
          this.notificationService.showSuccess('Password Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateStoreDetailsAction)
  updateStoreDetails(ctx: StateContext<AccountStateModel>, { payload }: UpdateStoreDetailsAction) {
    return this.accountService.updateStore(payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          const updateStore = { ...state.user.store, store: result };
          ctx.patchState({ ...state, user: updateStore });
        },
        complete: () => {
          this.notificationService.showSuccess('Store Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(AccountClearAction)
  accountClear(ctx: StateContext<AccountStateModel>) {
    ctx.patchState({
      user: null,
      permissions: [],
      roleName: null,
    });
  }
}
