import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetAttachmentsAction,
  CreateAttachmentAction,
  DeleteAttachmentAction,
  DeleteAllAttachmentAction,
} from '../action/attachment.action';
import { IAttachmentModel } from '../interface/attachment.interface';
import { AttachmentService } from '../services/attachment.service';
import { NotificationService } from '../services/notification.service';

export class AttachmentStateModel {
  attachment: IAttachmentModel;
}

@State<AttachmentStateModel>({
  name: 'attachment',
  defaults: {
    attachment: {
      data: [],
      total: 0,
    },
  },
})
@Injectable()
export class AttachmentState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private attachmentService = inject(AttachmentService);

  @Selector()
  static attachment(state: AttachmentStateModel) {
    return state.attachment;
  }

  @Action(GetAttachmentsAction)
  getAttachments(ctx: StateContext<AttachmentStateModel>, action: GetAttachmentsAction) {
    return this.attachmentService.getAttachments(action.payload).pipe(
      tap({
        next: result => {
          if (result) {
            ctx.patchState({
              attachment: result,
            });
          }
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateAttachmentAction)
  create(_ctx: StateContext<AttachmentStateModel>, action: CreateAttachmentAction) {
    return this.attachmentService.createAttachment(action.payload).pipe(
      tap({
        next: _result => {
          this.store.dispatch(new GetAttachmentsAction({ page: 1, paginate: 40 }));
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAttachmentAction)
  delete(_ctx: StateContext<AttachmentStateModel>, { id }: DeleteAttachmentAction) {
    return this.attachmentService.deleteAttachment(id).pipe(
      tap({
        next: _result => {
          this.store.dispatch(new GetAttachmentsAction({ page: 1, paginate: 40 }));
        },
        complete: () => {
          this.notificationService.showSuccess('IAttachment Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllAttachmentAction)
  deleteAll(_ctx: StateContext<AttachmentStateModel>, { ids }: DeleteAllAttachmentAction) {
    return this.attachmentService.deleteAllAttachment(ids).pipe(
      tap({
        next: _result => {
          this.store.dispatch(new GetAttachmentsAction({ page: 1, paginate: 40 }));
        },
        complete: () => {
          this.notificationService.showSuccess('IAttachment Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
