import { Injectable, inject } from '@angular/core';
import { Router } from '@angular/router';

import { Store, State, Selector, Action, StateContext } from '@ngxs/store';
import { tap } from 'rxjs/operators';

import { AccountClearAction } from '../action/account.action';
import {
  ForgotPassWordAction,
  LoginAction,
  VerifyEmailOtpAction,
  UpdatePasswordAction,
  LogoutAction,
  AuthClearAction,
} from '../action/auth.action';
import { GetNotificationAction } from '../action/notification.action';
import { GetSettingOptionAction } from '../action/setting.action';
import { GetBadgesAction } from '../action/sidebar.action';
import { AuthService } from '../services/auth.service';
import { NotificationService } from '../services/notification.service';

export interface AuthStateModel {
  email: string;
  token: string | number;
  access_token: string | null;
  permissions: [];
}

@State<AuthStateModel>({
  name: 'auth',
  defaults: {
    email: '',
    token: '',
    access_token: '',
    permissions: [],
  },
})
@Injectable()
export class AuthState {
  private store = inject(Store);
  router = inject(Router);
  private notificationService = inject(NotificationService);
  private authService = inject(AuthService);

  @Selector()
  static accessToken(state: AuthStateModel) {
    return state.access_token;
  }

  @Selector()
  static isAuthenticated(state: AuthStateModel) {
    return !!state.access_token;
  }

  @Selector()
  static email(state: AuthStateModel) {
    return state.email;
  }

  @Selector()
  static token(state: AuthStateModel) {
    return state.token;
  }

  @Action(LoginAction)
  login(ctx: StateContext<AuthStateModel>, action: LoginAction) {
    this.notificationService.notification = false;
    return this.authService.login(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            access_token: result.access_token,
            permissions: [],
          });
          this.store.dispatch(new GetBadgesAction());
          this.store.dispatch(new GetNotificationAction());
          this.store.dispatch(new GetSettingOptionAction());
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ForgotPassWordAction)
  forgotPassword(ctx: StateContext<AuthStateModel>, action: ForgotPassWordAction) {
    this.notificationService.notification = false;
    return this.authService.forgotPassword(action.payload).pipe(
      tap({
        next: () => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            email: action.payload.email,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(VerifyEmailOtpAction)
  verifyEmail(ctx: StateContext<AuthStateModel>, action: VerifyEmailOtpAction) {
    this.notificationService.notification = false;
    return this.authService.verifyEmailOtp(action.payload).pipe(
      tap({
        next: () => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            token: action.payload.token,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdatePasswordAction)
  updatePassword(_ctx: StateContext<AuthStateModel>, action: UpdatePasswordAction) {
    this.notificationService.notification = false;
    return this.authService.updatePassword(action.payload).pipe(
      tap({
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(LogoutAction)
  logout(_ctx: StateContext<AuthStateModel>) {
    this.notificationService.notification = false;
    return this.authService.logout().pipe(
      tap({
        next: _result => {
          this.store.dispatch(new AuthClearAction()).subscribe({
            complete: () => {
              void this.router.navigate(['/auth/login']);
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(AuthClearAction)
  authClear(ctx: StateContext<AuthStateModel>) {
    ctx.patchState({
      email: '',
      token: '',
      access_token: null,
      permissions: [],
    });
    this.store.dispatch(new AccountClearAction());
  }
}
