import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetAuthorsAction,
  CreateAuthorAction,
  EditAuthorAction,
  UpdateAuthorAction,
  UpdateAuthorStatusAction,
  DeleteAuthorAction,
  DeleteAllAuthorAction,
} from '../action/author.action';
import { IAuthor } from '../interface/author.interface';
import { AuthorService } from '../services/author.service';
import { NotificationService } from '../services/notification.service';

export class AuthorStateModel {
  author = {
    data: [] as IAuthor[],
    total: 0,
  };
  selectedAuthor: IAuthor | null;
}

@State<AuthorStateModel>({
  name: 'author',
  defaults: {
    author: {
      data: [],
      total: 0,
    },
    selectedAuthor: null,
  },
})
@Injectable()
export class AuthorState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private authorService = inject(AuthorService);

  @Selector()
  static author(state: AuthorStateModel) {
    return state.author;
  }

  @Selector()
  static selectedAuthor(state: AuthorStateModel) {
    return state.selectedAuthor;
  }

  @Selector()
  static authors(state: AuthorStateModel) {
    return state.author.data.map(author => {
      return { label: author?.author_name, value: author?.id };
    });
  }

  @Action(GetAuthorsAction)
  getAuthors(ctx: StateContext<AuthorStateModel>, action: GetAuthorsAction) {
    return this.authorService.getAuthors(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            author: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateAuthorAction)
  create(ctx: StateContext<AuthorStateModel>, action: CreateAuthorAction) {
    return this.authorService.createAuthor(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            author: {
              data: [...state.author.data, result],
              total: state?.author.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Author Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditAuthorAction)
  edit(ctx: StateContext<AuthorStateModel>, { id }: EditAuthorAction) {
    return this.authorService.editAuthor(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedAuthor: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateAuthorAction)
  update(ctx: StateContext<AuthorStateModel>, { payload, id }: UpdateAuthorAction) {
    return this.authorService.updateAuthor(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const authors = [...state.author.data];
            const index = authors.findIndex(author => author.id === id);
            authors[index] = result;

            ctx.patchState({
              ...state,
              author: {
                data: authors,
                total: state.author.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Author Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateAuthorStatusAction)
  updateStatus(ctx: StateContext<AuthorStateModel>, { id, status }: UpdateAuthorStatusAction) {
    return this.authorService.updateAuthorStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const authors = [...state.author.data];
            const index = authors.findIndex(author => author.id === id);
            authors[index] = result;

            ctx.patchState({
              ...state,
              author: {
                data: authors,
                total: state.author.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Author Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAuthorAction)
  delete(ctx: StateContext<AuthorStateModel>, { id }: DeleteAuthorAction) {
    return this.authorService.deleteAuthor(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetAuthorsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Author Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllAuthorAction)
  deleteAll(ctx: StateContext<AuthorStateModel>, { ids }: DeleteAllAuthorAction) {
    return this.authorService.deleteAllAuthor(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetAuthorsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Author Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
