import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetBlogsAction,
  CreateBlogAction,
  EditBlogAction,
  UpdateBlogAction,
  UpdateBlogStatusAction,
  DeleteBlogAction,
  DeleteAllBlogAction,
} from '../action/blog.action';
import { IBlog } from '../interface/blog.interface';
import { BlogService } from '../services/blog.service';
import { NotificationService } from '../services/notification.service';

export class BlogStateModel {
  blog = {
    data: [] as IBlog[],
    total: 0,
  };
  selectedBlog: IBlog | null;
}

@State<BlogStateModel>({
  name: 'blog',
  defaults: {
    blog: {
      data: [],
      total: 0,
    },
    selectedBlog: null,
  },
})
@Injectable()
export class BlogState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private blogService = inject(BlogService);

  @Selector()
  static blog(state: BlogStateModel) {
    return state.blog;
  }

  @Selector()
  static blogs(state: BlogStateModel) {
    return state.blog.data.map(res => {
      return { label: res?.title, value: res?.id };
    });
  }

  @Selector()
  static selectedBlog(state: BlogStateModel) {
    return state.selectedBlog;
  }

  @Action(GetBlogsAction)
  getBlogs(ctx: StateContext<BlogStateModel>, action: GetBlogsAction) {
    return this.blogService.getBlogs(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            blog: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateBlogAction)
  create(ctx: StateContext<BlogStateModel>, action: CreateBlogAction) {
    return this.blogService.createBlog(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            blog: {
              data: [...state.blog.data, result],
              total: state?.blog.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Blog Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditBlogAction)
  edit(ctx: StateContext<BlogStateModel>, { id }: EditBlogAction) {
    return this.blogService.editBlog(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedBlog: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateBlogAction)
  update(ctx: StateContext<BlogStateModel>, { payload, id }: UpdateBlogAction) {
    return this.blogService.updateBlog(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const blogs = [...state.blog.data];
            const index = blogs.findIndex(blog => blog.id === id);
            blogs[index] = result;

            ctx.patchState({
              ...state,
              blog: {
                data: blogs,
                total: state.blog.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Blog Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateBlogStatusAction)
  updateStatus(ctx: StateContext<BlogStateModel>, { id, status }: UpdateBlogStatusAction) {
    return this.blogService.updateBlogStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const blogs = [...state.blog.data];
            const index = blogs.findIndex(blog => blog.id === id);
            blogs[index] = result;

            ctx.patchState({
              ...state,
              blog: {
                data: blogs,
                total: state.blog.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Blog Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteBlogAction)
  delete(ctx: StateContext<BlogStateModel>, { id }: DeleteBlogAction) {
    return this.blogService.deleteBlog(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetBlogsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Blog Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllBlogAction)
  deleteAll(ctx: StateContext<BlogStateModel>, { ids }: DeleteAllBlogAction) {
    return this.blogService.deleteAllBlog(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetBlogsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Blog Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
