import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetBrandsAction,
  CreateBrandAction,
  EditBrandAction,
  UpdateBrandAction,
  UpdateBrandStatusAction,
  DeleteBrandAction,
  DeleteAllBrandAction,
  ExportBrandAction,
  ImportBrandAction,
} from '../action/brand.action';
import { IBrand } from '../interface/brand.interface';
import { BrandService } from '../services/brand.service';
import { NotificationService } from '../services/notification.service';

export class BrandStateModel {
  brand = {
    data: [] as IBrand[],
    total: 0,
  };
  selectedBrand: IBrand | null;
}

@State<BrandStateModel>({
  name: 'brand',
  defaults: {
    brand: {
      data: [],
      total: 0,
    },
    selectedBrand: null,
  },
})
@Injectable()
export class BrandState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private brandService = inject(BrandService);

  @Selector()
  static brand(state: BrandStateModel) {
    return state.brand;
  }

  @Selector()
  static selectedBrand(state: BrandStateModel) {
    return state.selectedBrand;
  }

  @Selector()
  static brands(state: BrandStateModel) {
    return state.brand.data.map(res => {
      return { label: res?.name, value: res?.id };
    });
  }

  @Action(GetBrandsAction)
  getBrands(ctx: StateContext<BrandStateModel>, action: GetBrandsAction) {
    return this.brandService.getBrands(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            brand: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateBrandAction)
  create(ctx: StateContext<BrandStateModel>, action: CreateBrandAction) {
    return this.brandService.createBrand(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            brand: {
              data: [...state.brand.data, result],
              total: state?.brand.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Brand Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditBrandAction)
  edit(ctx: StateContext<BrandStateModel>, { id }: EditBrandAction) {
    return this.brandService.editBrand(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedBrand: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateBrandAction)
  update(ctx: StateContext<BrandStateModel>, { payload, id }: UpdateBrandAction) {
    return this.brandService.updateBrand(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const brands = [...state.brand.data];
            const index = brands.findIndex(brand => brand.id === id);
            brands[index] = result;

            ctx.patchState({
              ...state,
              brand: {
                data: brands,
                total: state.brand.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Brand Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateBrandStatusAction)
  updateStatus(ctx: StateContext<BrandStateModel>, { id, status }: UpdateBrandStatusAction) {
    return this.brandService.updateBrandStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const brands = [...state.brand.data];
            const index = brands.findIndex(brand => brand.id === id);
            brands[index] = result;

            ctx.patchState({
              ...state,
              brand: {
                data: brands,
                total: state.brand.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Brand Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteBrandAction)
  delete(_ctx: StateContext<BrandStateModel>, { id }: DeleteBrandAction) {
    return this.brandService.deleteBrand(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetBrandsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Brans Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllBrandAction)
  deleteAll(_ctx: StateContext<BrandStateModel>, { ids }: DeleteAllBrandAction) {
    return this.brandService.deleteAllBrand(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetBrandsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Brand Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ImportBrandAction)
  import(_ctx: StateContext<BrandStateModel>, action: ImportBrandAction) {
    return this.brandService.importBrand(action.payload).pipe(
      tap({
        next: _result => {
          this.store.dispatch(new GetBrandsAction({ page: 1, paginate: 15 }));
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ExportBrandAction)
  export(_ctx: StateContext<BrandStateModel>, _action: ExportBrandAction) {
    return this.brandService.exportBrand().pipe(
      tap({
        next: result => {
          const blob = new Blob([result], { type: 'text/csv' });
          const url = window.URL.createObjectURL(blob);
          const link = document.createElement('a');
          link.href = url;
          link.download = 'brand.csv';
          link.click();
          window.URL.revokeObjectURL(url);
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
