import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { of, tap } from 'rxjs';

import {
  GetCartItemsAction,
  AddToCartAction,
  UpdateCartAction,
  DeleteCartAction,
  ClearCartAction,
} from '../action/cart.action';
import { ICart } from '../interface/cart.interface';
import { CartService } from '../services/cart.service';
import { NotificationService } from '../services/notification.service';

export interface CartStateModel {
  is_digital_only: boolean | number | null;
  items: ICart[];
  total: number;
}

@State<CartStateModel>({
  name: 'cart',
  defaults: {
    is_digital_only: null,
    items: [],
    total: 0,
  },
})
@Injectable()
export class CartState {
  private cartService = inject(CartService);
  private notificationService = inject(NotificationService);
  private store = inject(Store);

  @Selector()
  static cartItems(state: CartStateModel) {
    return state.items;
  }

  @Selector()
  static cartTotal(state: CartStateModel) {
    return state.total;
  }

  @Selector()
  static cartHasDigital(state: CartStateModel) {
    return state.is_digital_only;
  }

  @Action(GetCartItemsAction)
  getCartItems(ctx: StateContext<CartStateModel>) {
    return this.cartService.getCartItems().pipe(
      tap({
        next: result => {
          ctx.patchState(result);
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(AddToCartAction)
  add(ctx: StateContext<CartStateModel>, action: AddToCartAction) {
    return this.cartService.addToCart(action.payload).pipe(
      tap({
        next: result => {
          if (result.items.length) {
            const state = ctx.getState();
            const cart = [...state.items];
            const index = cart.findIndex(item => item.id === result.items[0].id);

            let output = { ...state };

            if (index != -1) {
              cart[index].quantity = cart[index]?.quantity + action?.payload.quantity;

              if (cart[index].product?.wholesales?.length) {
                let wholesale =
                  cart[index].product.wholesales.find(
                    value =>
                      value.min_qty <= cart[index].quantity &&
                      value.max_qty >= cart[index].quantity,
                  ) || null;
                if (wholesale && cart[index].product.wholesale_price_type == 'fixed') {
                  cart[index].sub_total = cart[index].quantity * wholesale.value;
                  cart[index].wholesale_price = cart[index].sub_total / cart[index].quantity;
                } else if (wholesale && cart[index].product.wholesale_price_type == 'percentage') {
                  cart[index].sub_total =
                    cart[index].quantity *
                    (cart[index]?.variation
                      ? cart[index]?.variation?.sale_price
                      : cart[index].product.sale_price);
                  cart[index].sub_total =
                    cart[index].sub_total - cart[index].sub_total * (wholesale.value / 100);
                  cart[index].wholesale_price = cart[index].sub_total / cart[index].quantity;
                } else {
                  cart[index].sub_total =
                    cart[index]?.quantity *
                    (cart[index]?.variation
                      ? cart[index]?.variation?.sale_price
                      : cart[index].product.sale_price);
                  cart[index].wholesale_price = null;
                }
              } else {
                cart[index].sub_total =
                  cart[index]?.quantity *
                  (cart[index]?.variation
                    ? cart[index]?.variation?.sale_price
                    : cart[index].product.sale_price);
                cart[index].wholesale_price = null;
              }
            } else {
              output.items = [...state.items, ...result.items];
            }

            // Calculate Total
            output.total = output.items.reduce((prev, curr: ICart) => {
              return prev + Number(curr.sub_total);
            }, 0);

            output.is_digital_only = result.is_digital_only;
            ctx.patchState(output);
          }
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateCartAction)
  update(ctx: StateContext<CartStateModel>, action: UpdateCartAction) {
    const state = ctx.getState();
    const cart = [...state.items];
    const index = cart.findIndex(item => item.id === action.payload.id);
    const productQty = cart[index]?.variation
      ? cart[index]?.variation?.quantity
      : cart[index]?.product?.quantity;

    if (productQty < cart[index]?.quantity + action?.payload.quantity) {
      this.notificationService.showError(
        `You cannot add more than ${productQty} items available in stock.`,
      );
      return false;
    }

    cart[index].quantity = cart[index]?.quantity + action?.payload.quantity;

    if (cart[index].product?.wholesales?.length) {
      let wholesale =
        cart[index].product.wholesales.find(
          value => value.min_qty <= cart[index].quantity && value.max_qty >= cart[index].quantity,
        ) || null;
      if (wholesale && cart[index].product.wholesale_price_type == 'fixed') {
        cart[index].sub_total = cart[index].quantity * wholesale.value;
        cart[index].wholesale_price = cart[index].sub_total / cart[index].quantity;
      } else if (wholesale && cart[index].product.wholesale_price_type == 'percentage') {
        cart[index].sub_total =
          cart[index].quantity *
          (cart[index]?.variation
            ? cart[index]?.variation?.sale_price
            : cart[index].product.sale_price);
        cart[index].sub_total =
          cart[index].sub_total - cart[index].sub_total * (wholesale.value / 100);
        cart[index].wholesale_price = cart[index].sub_total / cart[index].quantity;
      } else {
        cart[index].sub_total =
          cart[index]?.quantity *
          (cart[index]?.variation
            ? cart[index]?.variation?.sale_price
            : cart[index].product.sale_price);
        cart[index].wholesale_price = null;
      }
    } else {
      cart[index].sub_total =
        cart[index]?.quantity *
        (cart[index]?.variation
          ? cart[index]?.variation?.sale_price
          : cart[index].product.sale_price);
      cart[index].wholesale_price = null;
    }

    if (cart[index].quantity < 1) {
      this.store.dispatch(new DeleteCartAction(action.payload.id!));
      return of();
    }

    let total = state.items.reduce((prev, curr: ICart) => {
      return prev + Number(curr.sub_total);
    }, 0);

    ctx.patchState({
      ...state,
      total: total,
    });

    return this.cartService.updateCart(action.payload).pipe(
      tap({
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteCartAction)
  delete(ctx: StateContext<CartStateModel>, { id }: DeleteCartAction) {
    const state = ctx.getState();
    let cart = state.items.filter(value => value.id !== id);
    let total = state.items.reduce((prev, curr: ICart) => {
      return prev + Number(curr.sub_total);
    }, 0);
    ctx.patchState({
      items: cart,
      total: total,
    });
    return this.cartService.deleteCart(id).pipe(
      tap({
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ClearCartAction)
  clearCart(ctx: StateContext<CartStateModel>) {
    ctx.patchState({
      items: [],
      total: 0,
    });
  }
}
