import { Injectable, inject } from '@angular/core';
import { Router } from '@angular/router';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetCategoriesAction,
  CreateCategoryAction,
  EditCategoryAction,
  UpdateCategoryAction,
  DeleteCategoryAction,
  ExportCategoryAction,
  ImportCategoryAction,
} from '../action/category.action';
import { ICategory } from '../interface/category.interface';
import { CategoryService } from '../services/category.service';
import { NotificationService } from '../services/notification.service';

export class CategoryStateModel {
  category = {
    data: [] as ICategory[],
    total: 0,
  };
  selectedCategory: ICategory | null;
}

@State<CategoryStateModel>({
  name: 'category',
  defaults: {
    category: {
      data: [],
      total: 0,
    },
    selectedCategory: null,
  },
})
@Injectable()
export class CategoryState {
  private store = inject(Store);
  private router = inject(Router);
  private notificationService = inject(NotificationService);
  private categoryService = inject(CategoryService);

  @Selector()
  static category(state: CategoryStateModel) {
    return state.category;
  }

  @Selector()
  static categories(state: CategoryStateModel) {
    return state.category.data.map(res => {
      return {
        label: res?.name,
        value: res?.id,
        data: {
          name: res.name,
          slug: res.slug,
          image: res.category_icon ? res.category_icon.original_url : 'assets/images/category.png',
        },
      };
    });
  }

  @Selector()
  static categoriesSlug(state: CategoryStateModel) {
    return state.category.data.map(res => {
      return {
        label: res?.name,
        value: res?.slug,
        data: {
          name: res.name,
          slug: res.slug,
          image: res.category_icon ? res.category_icon.original_url : 'assets/images/category.png',
        },
      };
    });
  }

  @Selector()
  static selectedCategory(state: CategoryStateModel) {
    return state.selectedCategory;
  }

  @Action(GetCategoriesAction)
  getCategories(ctx: StateContext<CategoryStateModel>, action: GetCategoriesAction) {
    return this.categoryService.getCategories(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            category: {
              data: result.data,
              total: result?.total ? result?.total : result.data.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateCategoryAction)
  create(_ctx: StateContext<CategoryStateModel>, action: CreateCategoryAction) {
    return this.categoryService.createCategory(action.payload).pipe(
      tap({
        next: result => {
          this.store.dispatch(new GetCategoriesAction({ type: result?.type }));
        },
        complete: () => {
          this.notificationService.showSuccess('Category Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditCategoryAction)
  edit(ctx: StateContext<CategoryStateModel>, { id }: EditCategoryAction) {
    return this.categoryService.editCategory(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedCategory: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateCategoryAction)
  update(_ctx: StateContext<CategoryStateModel>, { payload, id }: UpdateCategoryAction) {
    return this.categoryService.updateCategory(id, payload).pipe(
      tap({
        next: _result => {
          this.store.dispatch(new GetCategoriesAction({ type: payload?.type }));
        },
        complete: () => {
          this.notificationService.showSuccess('Category Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteCategoryAction)
  delete(_ctx: StateContext<CategoryStateModel>, { id, type }: DeleteCategoryAction) {
    return this.categoryService.deleteCategory(id).pipe(
      tap({
        next: _result => {
          this.store.dispatch(new GetCategoriesAction({ type: type ? type : 'product' }));
          if (type == 'post') void this.router.navigateByUrl('/blog/category');
          else void this.router.navigateByUrl('/category');
        },
        complete: () => {
          this.notificationService.showSuccess('Category Deleted Successfully.');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ImportCategoryAction)
  import(_ctx: StateContext<CategoryStateModel>, action: ImportCategoryAction) {
    return this.categoryService.importCategory(action.payload).pipe(
      tap({
        next: _result => {
          this.store.dispatch(new GetCategoriesAction({ type: 'product' }));
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ExportCategoryAction)
  export(_ctx: StateContext<CategoryStateModel>, _action: ExportCategoryAction) {
    return this.categoryService.exportCategory().pipe(
      tap({
        next: result => {
          const blob = new Blob([result], { type: 'text/csv' });
          const url = window.URL.createObjectURL(blob);
          const link = document.createElement('a');
          link.href = url;
          link.download = 'category.csv';
          link.click();
          window.URL.revokeObjectURL(url);
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
