import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetCouponsAction,
  CreateCouponAction,
  EditCouponAction,
  UpdateCouponAction,
  UpdateCouponStatusAction,
  DeleteCouponAction,
  DeleteAllCouponAction,
} from '../action/coupon.action';
import { ICoupon } from '../interface/coupon.interface';
import { CouponService } from '../services/coupon.service';
import { NotificationService } from '../services/notification.service';

export class CouponStateModel {
  coupon = {
    data: [] as ICoupon[],
    total: 0,
  };
  selectedCoupon: ICoupon | null;
}

@State<CouponStateModel>({
  name: 'coupon',
  defaults: {
    coupon: {
      data: [],
      total: 0,
    },
    selectedCoupon: null,
  },
})
@Injectable()
export class CouponState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private couponService = inject(CouponService);

  @Selector()
  static coupon(state: CouponStateModel) {
    return state.coupon;
  }

  @Selector()
  static selectedCoupon(state: CouponStateModel) {
    return state.selectedCoupon;
  }

  @Selector()
  static coupons(state: CouponStateModel) {
    return state.coupon.data.map(coupon => {
      return { label: coupon?.title, value: coupon?.id };
    });
  }

  @Action(GetCouponsAction)
  getCoupons(ctx: StateContext<CouponStateModel>, action: GetCouponsAction) {
    return this.couponService.getCoupons(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            coupon: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateCouponAction)
  create(ctx: StateContext<CouponStateModel>, action: CreateCouponAction) {
    return this.couponService.createCoupon(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            coupon: {
              data: [...state.coupon.data, result],
              total: state?.coupon.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Coupon Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditCouponAction)
  edit(ctx: StateContext<CouponStateModel>, { id }: EditCouponAction) {
    return this.couponService.editCoupon(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedCoupon: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateCouponAction)
  update(ctx: StateContext<CouponStateModel>, { payload, id }: UpdateCouponAction) {
    return this.couponService.updateCoupon(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const coupons = [...state.coupon.data];
            const index = coupons.findIndex(coupon => coupon.id === id);
            coupons[index] = result;

            ctx.patchState({
              ...state,
              coupon: {
                data: coupons,
                total: state.coupon.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Coupon Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateCouponStatusAction)
  updateStatus(ctx: StateContext<CouponStateModel>, { id, status }: UpdateCouponStatusAction) {
    return this.couponService.updateCouponStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const coupons = [...state.coupon.data];
            const index = coupons.findIndex(coupon => coupon.id === id);
            coupons[index] = result;

            ctx.patchState({
              ...state,
              coupon: {
                data: coupons,
                total: state.coupon.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Coupon Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteCouponAction)
  delete(ctx: StateContext<CouponStateModel>, { id }: DeleteCouponAction) {
    return this.couponService.deleteCoupon(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetCouponsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Coupon Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllCouponAction)
  deleteAll(ctx: StateContext<CouponStateModel>, { ids }: DeleteAllCouponAction) {
    return this.couponService.deleteAllCoupon(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetCouponsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Coupon Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
