import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetCurrenciesAction,
  CreateCurrencyAction,
  EditCurrencyAction,
  UpdateCurrencyAction,
  UpdateCurrencyStatusAction,
  DeleteCurrencyAction,
  DeleteAllCurrencyAction,
} from '../action/currency.action';
import { ICurrency } from '../interface/currency.interface';
import { CurrencyService } from '../services/currency.service';
import { NotificationService } from '../services/notification.service';

export class CurrencyStateModel {
  currency = {
    data: [] as ICurrency[],
    total: 0,
  };
  selectedCurrency: ICurrency | null;
}

@State<CurrencyStateModel>({
  name: 'currency',
  defaults: {
    currency: {
      data: [],
      total: 0,
    },
    selectedCurrency: null,
  },
})
@Injectable()
export class CurrencyState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private currencyService = inject(CurrencyService);

  @Selector()
  static currency(state: CurrencyStateModel) {
    return state.currency;
  }

  @Selector()
  static currencies(state: CurrencyStateModel) {
    return state.currency.data.map(res => {
      return { label: res?.code, value: res?.id };
    });
  }

  @Selector()
  static selectedCurrency(state: CurrencyStateModel) {
    return state.selectedCurrency;
  }

  @Action(GetCurrenciesAction)
  getCurrencies(ctx: StateContext<CurrencyStateModel>, action: GetCurrenciesAction) {
    return this.currencyService.getCurrencies(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            currency: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateCurrencyAction)
  create(ctx: StateContext<CurrencyStateModel>, action: CreateCurrencyAction) {
    return this.currencyService.createCurrency(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            currency: {
              data: [...state.currency.data, result],
              total: state?.currency.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Currency Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditCurrencyAction)
  edit(ctx: StateContext<CurrencyStateModel>, { id }: EditCurrencyAction) {
    return this.currencyService.editCurrency(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedCurrency: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateCurrencyAction)
  update(ctx: StateContext<CurrencyStateModel>, { payload, id }: UpdateCurrencyAction) {
    return this.currencyService.updateCurrency(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const currencies = [...state.currency.data];
            const index = currencies.findIndex(currency => currency.id === id);
            currencies[index] = result;

            ctx.patchState({
              ...state,
              currency: {
                data: currencies,
                total: state.currency.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Currency Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateCurrencyStatusAction)
  updateStatus(ctx: StateContext<CurrencyStateModel>, { id, status }: UpdateCurrencyStatusAction) {
    return this.currencyService.updateCurrencyStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const currencies = [...state.currency.data];
            const index = currencies.findIndex(currency => currency.id === id);
            currencies[index] = result;

            ctx.patchState({
              ...state,
              currency: {
                data: currencies,
                total: state.currency.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Currency Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteCurrencyAction)
  delete(ctx: StateContext<CurrencyStateModel>, { id }: DeleteCurrencyAction) {
    return this.currencyService.deleteCurrency(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetCurrenciesAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Currency Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllCurrencyAction)
  deleteAll(ctx: StateContext<CurrencyStateModel>, { ids }: DeleteAllCurrencyAction) {
    return this.currencyService.deleteAllCurrency(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetCurrenciesAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Currency Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
