import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  CreateFaqAction,
  DeleteAllFaqAction,
  DeleteFaqAction,
  EditFaqAction,
  GetFaqsAction,
  UpdateFaqAction,
  UpdateFaqStatusAction,
} from '../action/faq.action';
import { IFaq } from '../interface/faq.interface';
import { FaqService } from '../services/faq.service';
import { NotificationService } from '../services/notification.service';

export class FaqStateModel {
  faq = {
    data: [] as IFaq[],
    total: 0,
  };
  selectedFaq: IFaq | null;
}

@State<FaqStateModel>({
  name: 'faq',
  defaults: {
    faq: {
      data: [],
      total: 0,
    },
    selectedFaq: null,
  },
})
@Injectable()
export class FaqState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private faqService = inject(FaqService);

  @Selector()
  static faq(state: FaqStateModel) {
    return state.faq;
  }

  @Selector()
  static selectedFaq(state: FaqStateModel) {
    return state.selectedFaq;
  }

  @Action(GetFaqsAction)
  getFaqs(ctx: StateContext<FaqStateModel>, action: GetFaqsAction) {
    return this.faqService.getFaqs(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            faq: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateFaqAction)
  create(ctx: StateContext<FaqStateModel>, action: CreateFaqAction) {
    return this.faqService.createFaq(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            faq: {
              data: [...state.faq.data, result],
              total: state?.faq.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Faq Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditFaqAction)
  edit(ctx: StateContext<FaqStateModel>, { id }: EditFaqAction) {
    return this.faqService.editFaq(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedFaq: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateFaqAction)
  update(ctx: StateContext<FaqStateModel>, { payload, id }: UpdateFaqAction) {
    return this.faqService.updateFaq(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const faqs = [...state.faq.data];
            const index = faqs.findIndex(faq => faq.id === id);
            faqs[index] = result;

            ctx.patchState({
              ...state,
              faq: {
                data: faqs,
                total: state.faq.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Faq Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateFaqStatusAction)
  updateStatus(ctx: StateContext<FaqStateModel>, { id, status }: UpdateFaqStatusAction) {
    return this.faqService.updateFaqStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const faqs = [...state.faq.data];
            const index = faqs.findIndex(faq => faq.id === id);
            faqs[index] = result;

            ctx.patchState({
              ...state,
              faq: {
                data: faqs,
                total: state.faq.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Faq Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteFaqAction)
  delete(ctx: StateContext<FaqStateModel>, { id }: DeleteFaqAction) {
    return this.faqService.deleteFaq(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetFaqsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Faq Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllFaqAction)
  deleteAll(ctx: StateContext<FaqStateModel>, { ids }: DeleteAllFaqAction) {
    return this.faqService.deleteAllFaq(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetFaqsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Faqs Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
