import { Injectable, inject } from '@angular/core';

import { TranslateService } from '@ngx-translate/core';
import { Action, Selector, State, StateContext, Store } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  CreateLanguageAction,
  DeleteLanguageAction,
  EditFormLanguageAction,
  EditLanguageAction,
  GetLanguageAction,
  GetLanguageFilesAction,
  GetLanguageFilesFieldsAction,
  LocalLanguageAction,
  UpdateLanguageAction,
  UpdateLanguageFilesFieldsAction,
  UpdateLanguageStatusAction,
} from '../action/language.action';
import { ILanguage } from '../interface/language.interface';
import { LanguageService } from '../services/language.service';
import { NavService } from '../services/nav.service';
import { NotificationService } from '../services/notification.service';
import { TranslationLoader } from '../services/translation-loader.service';

export class LanguageStateModel {
  language = {
    data: [] as ILanguage[],
    total: 0,
  };
  selectedLanguage: ILanguage | null;
  formLanguage: string;
  localLanguage: ILanguage | null;
  files: { name: string; slug: string }[];
  filesFields = {
    data: [] as { [key: string]: string }[],
    total: 0,
  };
}

@State<LanguageStateModel>({
  name: 'language',
  defaults: {
    language: {
      data: [],
      total: 0,
    },
    selectedLanguage: null,
    formLanguage: '',
    localLanguage: null,
    files: [],
    filesFields: {
      data: [],
      total: 0,
    },
  },
})
@Injectable()
export class LanguageState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private languageService = inject(LanguageService);
  private translationLoader = inject(TranslationLoader);
  private navServices = inject(NavService);
  private translate = inject(TranslateService);

  @Selector()
  static language(state: LanguageStateModel) {
    return state.language;
  }

  @Selector()
  static selectedLanguage(state: LanguageStateModel) {
    return state.selectedLanguage;
  }

  @Selector()
  static languages(state: LanguageStateModel) {
    return state.language.data.map(res => {
      return { label: res?.name, value: res?.id };
    });
  }

  @Selector()
  static languagesFiles(state: LanguageStateModel) {
    if (state.files)
      return state.files.map(res => {
        return { label: res?.name, value: res?.slug };
      });
  }

  @Selector()
  static languagesFilesFields(state: LanguageStateModel) {
    return state.filesFields;
  }

  @Selector()
  static localLanguage(state: LanguageStateModel) {
    return state.localLanguage;
  }

  @Selector()
  static formLanguage(state: LanguageStateModel) {
    return state.formLanguage;
  }

  @Action(GetLanguageAction)
  getLanguages(ctx: StateContext<LanguageStateModel>, action: GetLanguageAction) {
    this.navServices.sidebarLoading = true;
    return this.languageService.getLanguage(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            language: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
        complete: () => {
          this.navServices.sidebarLoading = false;
        },
      }),
    );
  }

  @Action(CreateLanguageAction)
  create(ctx: StateContext<LanguageStateModel>, action: CreateLanguageAction) {
    return this.languageService.createLanguage(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            language: {
              data: [...state.language.data, result],
              total: state?.language.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Language Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditLanguageAction)
  edit(ctx: StateContext<LanguageStateModel>, { id }: EditLanguageAction) {
    return this.languageService.editLanguage(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedLanguage: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateLanguageAction)
  update(ctx: StateContext<LanguageStateModel>, { payload, id }: UpdateLanguageAction) {
    return this.languageService.updateLanguage(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const languages = [...state.language.data];
            const index = languages.findIndex(language => language.id === id);
            languages[index] = result;

            ctx.patchState({
              ...state,
              language: {
                data: languages,
                total: state.language.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Language Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateLanguageStatusAction)
  updateStatus(ctx: StateContext<LanguageStateModel>, { id, status }: UpdateLanguageStatusAction) {
    return this.languageService.updateLanguageStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const languages = [...state.language.data];
            const index = languages.findIndex(language => language.id === id);
            languages[index] = result;

            ctx.patchState({
              ...state,
              language: {
                data: languages,
                total: state.language.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Language Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteLanguageAction)
  delete(ctx: StateContext<LanguageStateModel>, { id }: DeleteLanguageAction) {
    return this.languageService.deleteLanguage(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetLanguageAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Language Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditFormLanguageAction)
  formLanguage(ctx: StateContext<LanguageStateModel>, { lang }: EditFormLanguageAction) {
    const state = ctx.getState();
    ctx.patchState({
      ...state,
      formLanguage: lang,
    });
    this.translationLoader.getTranslation();
  }

  @Action(LocalLanguageAction)
  localLanguage(ctx: StateContext<LanguageStateModel>, { language }: LocalLanguageAction) {
    const state = ctx.getState();
    ctx.patchState({
      ...state,
      localLanguage: language,
    });
    this.translate.use(language.locale);
  }

  @Action(GetLanguageFilesAction)
  getLanguagesFiles(ctx: StateContext<LanguageStateModel>) {
    return this.languageService.getLanguageFiles().pipe(
      tap({
        next: result => {
          ctx.patchState({
            files: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(GetLanguageFilesFieldsAction)
  getLanguagesFilesFields(
    ctx: StateContext<LanguageStateModel>,
    action: GetLanguageFilesFieldsAction,
  ) {
    return this.languageService.getLanguageFilesFields(action.file, action.params).pipe(
      tap({
        next: result => {
          ctx.patchState({
            filesFields: {
              data: result.data,
              total: result.total,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateLanguageFilesFieldsAction)
  updateLanguagesFilesFields(
    ctx: StateContext<LanguageStateModel>,
    action: UpdateLanguageFilesFieldsAction,
  ) {
    return this.languageService
      .updateLanguageFilesFields(action.file, action.payload, action.params)
      .pipe(
        tap({
          next: result => {
            ctx.patchState({
              filesFields: {
                data: result.data,
                total: result.total,
              },
            });
            this.store.dispatch(new GetLanguageFilesAction());
          },
          error: err => {
            throw new Error(err?.error?.message);
          },
        }),
      );
  }
}
