import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetLicenseKeysAction,
  CreateLicenseKeyAction,
  EditLicenseKeyAction,
  UpdateLicenseKeyAction,
  UpdateLicenseKeyStatusAction,
  DeleteLicenseKeyAction,
  DeleteAllLicenseKeyAction,
} from '../action/license-key.actions';
import { ILicenseKey } from '../interface/license-key.interface';
import { LicenseKeyService } from '../services/license-key.service';
import { NotificationService } from '../services/notification.service';

export class LicenseKeyStateModel {
  licenseKey = {
    data: [] as ILicenseKey[],
    total: 0,
  };
  selectedLicenseKey: ILicenseKey | null;
}

@State<LicenseKeyStateModel>({
  name: 'licenseKey',
  defaults: {
    licenseKey: {
      data: [],
      total: 0,
    },
    selectedLicenseKey: null,
  },
})
@Injectable()
export class LicenseKeysState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private licenseKeyServiceService = inject(LicenseKeyService);

  @Selector()
  static licenseKey(state: LicenseKeyStateModel) {
    return state.licenseKey;
  }

  @Selector()
  static selectedLicenseKey(state: LicenseKeyStateModel) {
    return state.selectedLicenseKey;
  }

  @Action(GetLicenseKeysAction)
  getLicenseKeys(ctx: StateContext<LicenseKeyStateModel>, action: GetLicenseKeysAction) {
    return this.licenseKeyServiceService.getLicenseKeys(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            licenseKey: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateLicenseKeyAction)
  create(ctx: StateContext<LicenseKeyStateModel>, action: CreateLicenseKeyAction) {
    return this.licenseKeyServiceService.createLicenseKey(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            licenseKey: {
              data: [...state.licenseKey.data, result],
              total: state?.licenseKey.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('License Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditLicenseKeyAction)
  edit(ctx: StateContext<LicenseKeyStateModel>, { id }: EditLicenseKeyAction) {
    return this.licenseKeyServiceService.editLicenseKey(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedLicenseKey: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateLicenseKeyAction)
  update(ctx: StateContext<LicenseKeyStateModel>, { payload, id }: UpdateLicenseKeyAction) {
    return this.licenseKeyServiceService.updateLicenseKey(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const licenseKeys = [...state.licenseKey.data];
            const index = licenseKeys.findIndex(faq => faq.id === id);
            licenseKeys[index] = result;

            ctx.patchState({
              ...state,
              licenseKey: {
                data: licenseKeys,
                total: state.licenseKey.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('License Key Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateLicenseKeyStatusAction)
  updateStatus(
    ctx: StateContext<LicenseKeyStateModel>,
    { id, status }: UpdateLicenseKeyStatusAction,
  ) {
    return this.licenseKeyServiceService.updateLicenseKeyStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const licenseKeys = [...state.licenseKey.data];
            const index = licenseKeys.findIndex(faq => faq.id === id);
            licenseKeys[index] = result;

            ctx.patchState({
              ...state,
              licenseKey: {
                data: licenseKeys,
                total: state.licenseKey.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('License Key Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteLicenseKeyAction)
  delete(ctx: StateContext<LicenseKeyStateModel>, { id }: DeleteLicenseKeyAction) {
    return this.licenseKeyServiceService.deleteLicenseKey(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetLicenseKeysAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('License Key Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllLicenseKeyAction)
  deleteAll(ctx: StateContext<LicenseKeyStateModel>, { ids }: DeleteAllLicenseKeyAction) {
    return this.licenseKeyServiceService.deleteAllLicenseKey(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetLicenseKeysAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('License Key Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
