import { Injectable, inject } from '@angular/core';
import { Router } from '@angular/router';

import { Action, Selector, State, StateContext, Store } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  CreateMenuAction,
  DeleteMenuAction,
  EditMenuAction,
  GetMenuAction,
  UpdateMenuAction,
  UpdateSortMenuAction,
} from '../action/menu.action';
import { IMenu } from '../interface/menu.interface';
import { MenuService } from '../services/menu.service';
import { NotificationService } from '../services/notification.service';

export class MenuStateModel {
  menu = {
    data: [] as IMenu[],
    total: 0,
  };
  selectedMenu: IMenu | null;
}

@State<MenuStateModel>({
  name: 'menu',
  defaults: {
    menu: {
      data: [],
      total: 0,
    },
    selectedMenu: null,
  },
})
@Injectable()
export class MenuState {
  private notificationService = inject(NotificationService);
  private menuService = inject(MenuService);
  private store = inject(Store);
  private router = inject(Router);

  @Selector()
  static menu(state: MenuStateModel) {
    return state.menu;
  }

  @Selector()
  static selectedMenu(state: MenuStateModel) {
    return state.selectedMenu;
  }

  @Action(GetMenuAction)
  getMenu(ctx: StateContext<MenuStateModel>, action: GetMenuAction) {
    return this.menuService.getMenu(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            menu: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateMenuAction)
  create(_ctx: StateContext<MenuStateModel>, action: CreateMenuAction) {
    return this.menuService.createMenu(action.payload).pipe(
      tap({
        complete: () => {
          this.store.dispatch(new GetMenuAction());
          this.notificationService.showSuccess('Menu Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditMenuAction)
  edit(ctx: StateContext<MenuStateModel>, { id }: EditMenuAction) {
    return this.menuService.editMenu(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedMenu: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateMenuAction)
  update(_ctx: StateContext<MenuStateModel>, { payload, id }: UpdateMenuAction) {
    return this.menuService.updateMenu(id, payload).pipe(
      tap({
        complete: () => {
          this.notificationService.showSuccess('Menu Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateSortMenuAction)
  updateShort(ctx: StateContext<MenuStateModel>, action: UpdateSortMenuAction) {
    return this.menuService.updateSortMenu(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            menu: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Menu Sequence Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteMenuAction)
  delete(_ctx: StateContext<MenuStateModel>, { id }: DeleteMenuAction) {
    return this.menuService.deleteMenu(id).pipe(
      tap({
        next: _result => {
          this.store.dispatch(new GetMenuAction());
          void this.router.navigateByUrl('/menu');
        },
        complete: () => {
          this.notificationService.showSuccess('Menu Deleted Successfully.');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
