import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  CreateNoticeAction,
  DeleteNoticeAction,
  EditNoticeAction,
  GetNoticeAction,
  MarkAsReadNoticeAction,
  ResentNoticeAction,
  UpdateNoticeAction,
} from '../action/notice.action';
import { INotice } from '../interface/notice.interface';
import { NoticeService } from '../services/notice.service';
import { NotificationService } from '../services/notification.service';

export class NoticeStateModel {
  notice = {
    data: [] as INotice[],
    total: 0,
  };
  selectedNotice: INotice | null;
  recentNotice: INotice | null;
}

@State<NoticeStateModel>({
  name: 'notice',
  defaults: {
    notice: {
      data: [],
      total: 0,
    },
    selectedNotice: null,
    recentNotice: null,
  },
})
@Injectable()
export class NoticeState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private noticeService = inject(NoticeService);

  @Selector()
  static notice(state: NoticeStateModel) {
    return state.notice;
  }

  @Selector()
  static selectedNotice(state: NoticeStateModel) {
    return state.selectedNotice;
  }

  @Selector()
  static recentNotice(state: NoticeStateModel) {
    return state.recentNotice;
  }

  @Action(GetNoticeAction)
  getNotices(ctx: StateContext<NoticeStateModel>, action: GetNoticeAction) {
    return this.noticeService.getNotice(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            notice: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateNoticeAction)
  create(ctx: StateContext<NoticeStateModel>, action: CreateNoticeAction) {
    return this.noticeService.createNotice(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            notice: {
              data: [...state.notice.data, result],
              total: state?.notice.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Notice Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditNoticeAction)
  edit(ctx: StateContext<NoticeStateModel>, { id }: EditNoticeAction) {
    return this.noticeService.editNotice(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedNotice: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ResentNoticeAction)
  recentNotice(ctx: StateContext<NoticeStateModel>, { id }: ResentNoticeAction) {
    return this.noticeService.editNotice(id).pipe(
      tap({
        next: result => {
          ctx.patchState({
            recentNotice: result?.is_read === 0 ? result : null,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateNoticeAction)
  update(ctx: StateContext<NoticeStateModel>, { payload, id }: UpdateNoticeAction) {
    return this.noticeService.updateNotice(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const notices = [...state.notice.data];
            const index = notices.findIndex(notice => notice.id === id);
            notices[index] = result;

            ctx.patchState({
              ...state,
              notice: {
                data: notices,
                total: state.notice.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Notice Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(MarkAsReadNoticeAction)
  markAsRead(ctx: StateContext<NoticeStateModel>, { id }: MarkAsReadNoticeAction) {
    return this.noticeService.markAsRead(id).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const notices = [...state.notice.data];
            const index = notices.findIndex(notice => notice.id === id);
            notices[index] = result;

            ctx.patchState({
              ...state,
              notice: {
                data: notices,
                total: state.notice.total,
              },
              recentNotice: null,
            });
          }
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteNoticeAction)
  delete(ctx: StateContext<NoticeStateModel>, { id }: DeleteNoticeAction) {
    return this.noticeService.deleteNotice(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetNoticeAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Notice Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
