import { Injectable, inject } from '@angular/core';
import { Router } from '@angular/router';

import { Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetOrdersAction,
  SelectUserAction,
  ViewOrderAction,
  CheckoutAction,
  PlaceOrderAction,
  UpdateOrderStatusAction,
  ClearAction,
  DownloadInvoiceAction,
} from '../action/order.action';
import { IOrder, IOrderCheckout } from '../interface/order.interface';
import { IUser } from '../interface/user.interface';
import { OrderService } from '../services/order.service';
import { UserService } from '../services/user.service';

export class OrderStateModel {
  order = {
    data: [] as IOrder[],
    total: 0,
  };
  selectedOrder: IOrder | null;
  selectedUser: IUser | null;
  checkout: IOrderCheckout | null;
}

@State<OrderStateModel>({
  name: 'order',
  defaults: {
    order: {
      data: [],
      total: 0,
    },
    selectedOrder: null,
    selectedUser: null,
    checkout: null,
  },
})
@Injectable()
export class OrderState {
  private router = inject(Router);
  private orderService = inject(OrderService);
  private userService = inject(UserService);

  @Selector()
  static order(state: OrderStateModel) {
    return state.order;
  }

  @Selector()
  static selectedUser(state: OrderStateModel) {
    return state.selectedUser;
  }

  @Selector()
  static selectedOrder(state: OrderStateModel) {
    return state.selectedOrder;
  }

  @Selector()
  static checkout(state: OrderStateModel) {
    return state.checkout;
  }

  @Action(GetOrdersAction)
  getOrders(ctx: StateContext<OrderStateModel>, action: GetOrdersAction) {
    return this.orderService.getOrders(action?.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            order: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(SelectUserAction)
  selectUser(ctx: StateContext<OrderStateModel>, { id }: SelectUserAction) {
    return this.userService.editUser(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedUser: result,
            checkout: null,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ViewOrderAction)
  viewOrder(ctx: StateContext<OrderStateModel>, { id }: ViewOrderAction) {
    return this.orderService.viewOrder(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedOrder: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CheckoutAction)
  checkout(ctx: StateContext<OrderStateModel>, action: CheckoutAction) {
    return this.orderService.checkout(action?.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            checkout: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(PlaceOrderAction)
  placeOrder(ctx: StateContext<OrderStateModel>, action: PlaceOrderAction) {
    return this.orderService.placeOrder(action?.payload).pipe(
      tap({
        next: result => {
          void this.router.navigateByUrl(`/order/details/${result.order_number}`);
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateOrderStatusAction)
  updateOrderStatus(ctx: StateContext<OrderStateModel>, { id, payload }: UpdateOrderStatusAction) {
    return this.orderService.updateOrderStatus(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const orders = [...state.order.data];
            const index = orders.findIndex(order => order.id === id);
            orders[index] = result;

            ctx.patchState({
              ...state,
              order: {
                data: orders,
                total: state.order.total,
              },
              selectedOrder: result,
            });
          }
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DownloadInvoiceAction)
  downloadInvoice(ctx: StateContext<OrderStateModel>, action: DownloadInvoiceAction) {
    return this.orderService.downloadInvoice(action.payload).pipe(
      tap({
        next: result => {
          const blob = new Blob([result], { type: 'pdf' });
          const url = window.URL.createObjectURL(blob);
          const link = document.createElement('a');
          link.href = url;
          link.download = `invoice-${action.payload['order_number']}.pdf`;
          link.click();
          window.URL.revokeObjectURL(url);
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ClearAction)
  clear(ctx: StateContext<OrderStateModel>) {
    const state = ctx.getState();
    ctx.patchState({
      ...state,
      selectedUser: null,
      checkout: null,
    });
  }
}
