import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetPagesAction,
  CreatePageAction,
  EditPageAction,
  UpdatePageAction,
  UpdatePageStatusAction,
  DeletePageAction,
  DeleteAllPageAction,
} from '../action/page.action';
import { IPage } from '../interface/page.interface';
import { NotificationService } from '../services/notification.service';
import { PageService } from '../services/page.service';

export class PageStateModel {
  page = {
    data: [] as IPage[],
    total: 0,
  };
  selectedPage: IPage | null;
}

@State<PageStateModel>({
  name: 'page',
  defaults: {
    page: {
      data: [],
      total: 0,
    },
    selectedPage: null,
  },
})
@Injectable()
export class PageState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private pageService = inject(PageService);

  @Selector()
  static page(state: PageStateModel) {
    return state.page;
  }

  @Selector()
  static pages(state: PageStateModel) {
    return state.page.data.map(res => {
      return { label: res?.title, value: res?.slug };
    });
  }

  @Selector()
  static selectedPage(state: PageStateModel) {
    return state.selectedPage;
  }

  @Action(GetPagesAction)
  getPages(ctx: StateContext<PageStateModel>, action: GetPagesAction) {
    return this.pageService.getPages(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            page: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreatePageAction)
  create(ctx: StateContext<PageStateModel>, action: CreatePageAction) {
    return this.pageService.createPage(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            page: {
              data: [...state.page.data, result],
              total: state?.page.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Page Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditPageAction)
  edit(ctx: StateContext<PageStateModel>, { id }: EditPageAction) {
    return this.pageService.editPage(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedPage: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdatePageAction)
  update(ctx: StateContext<PageStateModel>, { payload, id }: UpdatePageAction) {
    return this.pageService.updatePage(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const pages = [...state.page.data];
            const index = pages.findIndex(page => page.id === id);
            pages[index] = result;

            ctx.patchState({
              ...state,
              page: {
                data: pages,
                total: state.page.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Page Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdatePageStatusAction)
  updateStatus(ctx: StateContext<PageStateModel>, { id, status }: UpdatePageStatusAction) {
    return this.pageService.updatePageStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const pages = [...state.page.data];
            const index = pages.findIndex(page => page.id === id);
            pages[index] = result;

            ctx.patchState({
              ...state,
              page: {
                data: pages,
                total: state.page.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Page Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeletePageAction)
  delete(ctx: StateContext<PageStateModel>, { id }: DeletePageAction) {
    return this.pageService.deletePage(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetPagesAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Page Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllPageAction)
  deleteAll(ctx: StateContext<PageStateModel>, { ids }: DeleteAllPageAction) {
    return this.pageService.deleteAllPage(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetPagesAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Pages Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
