import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { Select2Data } from 'ng-select2-component';
import { tap } from 'rxjs';

import {
  GetProductsAction,
  CreateProductAction,
  EditProductAction,
  UpdateProductAction,
  UpdateProductStatusAction,
  ApproveProductStatusAction,
  DeleteProductAction,
  DeleteAllProductAction,
  ReplicateProductAction,
  ExportProductAction,
  ImportProductAction,
  DownloadAction,
} from '../action/product.action';
import { UpdateBadgeValueAction } from '../action/sidebar.action';
import { IProduct, IProductModel, IVariation } from '../interface/product.interface';
import { NotificationService } from '../services/notification.service';
import { ProductService } from '../services/product.service';

export class ProductStateModel {
  product = {
    data: [] as IProduct[],
    total: 0,
  };
  selectedProduct: IProduct | null;
  topSellingProducts: IProduct[];
}

@State<ProductStateModel>({
  name: 'product',
  defaults: {
    product: {
      data: [],
      total: 0,
    },
    selectedProduct: null,
    topSellingProducts: [],
  },
})
@Injectable()
export class ProductState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private productService = inject(ProductService);

  @Selector()
  static product(state: ProductStateModel) {
    return state.product;
  }

  @Selector()
  static products(state: ProductStateModel) {
    return state.product.data
      .filter(data => data.id !== state.selectedProduct?.id)
      .map((res: IProduct) => {
        return {
          label: res?.name,
          value: res?.id,
          data: {
            type: res.type,
            name: res.name,
            slug: res.slug,
            stock_status: res.stock_status,
            image: res.product_thumbnail
              ? res.product_thumbnail.original_url
              : 'assets/images/product.png',
          },
        };
      });
  }

  @Selector()
  static digitalProducts(state: ProductStateModel) {
    let products: Select2Data = [];
    state.product.data
      .filter(data => data.id !== state.selectedProduct?.id && data.product_type == 'digital')
      .map((res: IProduct) => {
        products.push({
          label: res?.name,
          value: res?.id,
          data: {
            name: res.name,
            product_id: res?.id,
            variation_id: null,
            image: res.product_thumbnail
              ? res.product_thumbnail.original_url
              : 'assets/images/product.png',
          },
        });
        if (res?.variations?.length) {
          res.variations.map((variation: IVariation) => {
            products.push({
              label: variation?.name,
              value: variation?.id!,
              data: {
                name: variation.name,
                product_id: res?.id,
                variation_id: variation?.id,
                image: variation.variation_image
                  ? variation.variation_image.original_url
                  : 'assets/images/product.png',
              },
            });
          });
        }
      });
    return products;
  }

  @Selector()
  static selectedProduct(state: ProductStateModel) {
    return state.selectedProduct;
  }

  @Selector()
  static topSellingProducts(state: ProductStateModel) {
    return state.topSellingProducts;
  }

  @Action(GetProductsAction)
  getProducts(ctx: StateContext<ProductStateModel>, action: GetProductsAction) {
    return this.productService.getProducts(action.payload).pipe(
      tap({
        next: (result: IProductModel) => {
          if (action?.payload!['top_selling']) {
            const state = ctx.getState();
            ctx.patchState({
              ...state,
              topSellingProducts: result.data,
            });
          } else {
            ctx.patchState({
              product: {
                data: result.data,
                total: result?.total ? result?.total : result.data?.length,
              },
            });
          }
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateProductAction)
  create(ctx: StateContext<ProductStateModel>, action: CreateProductAction) {
    return this.productService.createProduct(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            product: {
              data: [...state.product.data, result],
              total: state?.product.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Product Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditProductAction)
  edit(ctx: StateContext<ProductStateModel>, { id }: EditProductAction) {
    return this.productService.editProduct(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedProduct: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateProductAction)
  update(ctx: StateContext<ProductStateModel>, { payload, id }: UpdateProductAction) {
    return this.productService.updateProduct(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const products = [...state.product.data];
            const index = products.findIndex(product => product.id === id);
            products[index] = result;

            ctx.patchState({
              ...state,
              product: {
                data: products,
                total: state.product.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Product Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateProductStatusAction)
  updateStatus(ctx: StateContext<ProductStateModel>, { id, status }: UpdateProductStatusAction) {
    return this.productService.updateProductStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const products = [...state.product.data];
            const index = products.findIndex(product => product.id === id);
            products[index] = result;

            ctx.patchState({
              ...state,
              product: {
                data: products,
                total: state.product.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Product Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ApproveProductStatusAction)
  approveStatus(ctx: StateContext<ProductStateModel>, { id, status }: ApproveProductStatusAction) {
    return this.productService.approveProductStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const products = [...state.product.data];
            const index = products.findIndex(product => product.id === id);
            products[index] = result;
            ctx.patchState({
              ...state,
              product: {
                data: products,
                total: state.product.total,
              },
            });
            this.store.dispatch(
              new UpdateBadgeValueAction('/product', result?.total_in_approved_products),
            );
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Product Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteProductAction)
  delete(_ctx: StateContext<ProductStateModel>, { id }: DeleteProductAction) {
    return this.productService.deleteProduct(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetProductsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Product Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllProductAction)
  deleteAll(_ctx: StateContext<ProductStateModel>, { ids }: DeleteAllProductAction) {
    return this.productService.deleteAllProduct(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetProductsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Product Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ReplicateProductAction)
  replicateProduct(_ctx: StateContext<ProductStateModel>, { ids }: ReplicateProductAction) {
    return this.productService.replicateProduct(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetProductsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Product Replicated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ImportProductAction)
  import(_ctx: StateContext<ProductStateModel>, action: ImportProductAction) {
    return this.productService.importProduct(action.payload).pipe(
      tap({
        next: _result => {
          this.store.dispatch(new GetProductsAction({ page: 1, paginate: 15 }));
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(ExportProductAction)
  export(_ctx: StateContext<ProductStateModel>, action: ExportProductAction) {
    return this.productService.exportProduct(action.payload).pipe(
      tap({
        next: result => {
          const blob = new Blob([result], { type: 'text/csv' });
          const url = window.URL.createObjectURL(blob);
          const link = document.createElement('a');
          link.href = url;
          link.download = 'products.csv';
          link.click();
          window.URL.revokeObjectURL(url);
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DownloadAction)
  download(_ctx: StateContext<ProductStateModel>, action: DownloadAction) {
    return this.productService.download(action.payload).pipe(
      tap({
        next: result => {
          if (result && result.download_link) {
            window.location.assign(result.download_link);
          }
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
