import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetPublicationsAction,
  CreatePublicationAction,
  EditPublicationAction,
  UpdatePublicationAction,
  UpdatePublicationStatusAction,
  DeletePublicationAction,
  DeleteAllPublicationAction,
} from '../action/publication.action';
import { IPublication } from '../interface/publication.interface';
import { NotificationService } from '../services/notification.service';
import { PublicationService } from '../services/publication.service';

export class PublicationStateModel {
  publication = {
    data: [] as IPublication[],
    total: 0,
  };
  selectedPublication: IPublication | null;
}

@State<PublicationStateModel>({
  name: 'publication',
  defaults: {
    publication: {
      data: [],
      total: 0,
    },
    selectedPublication: null,
  },
})
@Injectable()
export class PublicationState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private publicationService = inject(PublicationService);

  @Selector()
  static publication(state: PublicationStateModel) {
    return state.publication;
  }

  @Selector()
  static selectedPublication(state: PublicationStateModel) {
    return state.selectedPublication;
  }

  @Selector()
  static publications(state: PublicationStateModel) {
    return state.publication.data.map(publication => {
      return { label: publication?.publisher_name, value: publication?.id };
    });
  }

  @Action(GetPublicationsAction)
  getPublications(ctx: StateContext<PublicationStateModel>, action: GetPublicationsAction) {
    return this.publicationService.getPublications(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            publication: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreatePublicationAction)
  create(ctx: StateContext<PublicationStateModel>, action: CreatePublicationAction) {
    return this.publicationService.createPublication(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            publication: {
              data: [...state.publication.data, result],
              total: state?.publication.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Publication Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditPublicationAction)
  edit(ctx: StateContext<PublicationStateModel>, { id }: EditPublicationAction) {
    return this.publicationService.editPublication(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedPublication: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdatePublicationAction)
  update(ctx: StateContext<PublicationStateModel>, { payload, id }: UpdatePublicationAction) {
    return this.publicationService.updatePublication(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const publications = [...state.publication.data];
            const index = publications.findIndex(publication => publication.id === id);
            publications[index] = result;

            ctx.patchState({
              ...state,
              publication: {
                data: publications,
                total: state.publication.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Publication Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdatePublicationStatusAction)
  updateStatus(
    ctx: StateContext<PublicationStateModel>,
    { id, status }: UpdatePublicationStatusAction,
  ) {
    return this.publicationService.updatePublicationStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const publications = [...state.publication.data];
            const index = publications.findIndex(publication => publication.id === id);
            publications[index] = result;

            ctx.patchState({
              ...state,
              publication: {
                data: publications,
                total: state.publication.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Publication Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeletePublicationAction)
  delete(ctx: StateContext<PublicationStateModel>, { id }: DeletePublicationAction) {
    return this.publicationService.deletePublication(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetPublicationsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Publication Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllPublicationAction)
  deleteAll(ctx: StateContext<PublicationStateModel>, { ids }: DeleteAllPublicationAction) {
    return this.publicationService.deleteAllPublication(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetPublicationsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Publication Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
