import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetQuestionAnswersAction,
  EditQuestionAnswersAction,
  UpdateQuestionAnswersAction,
  DeleteAllQuestionAnswersAction,
  DeleteQuestionAnswersAction,
} from '../action/questions-answers.action';
import { IQuestionAnswers } from '../interface/questions-answers.interface';
import { NotificationService } from '../services/notification.service';
import { QuestionsAnswersService } from '../services/questions-answers.service';

export class QuestionAnswersStateModel {
  question = {
    data: [] as IQuestionAnswers[],
    total: 0,
  };
  selectedQuestion: IQuestionAnswers | null;
}

@State<QuestionAnswersStateModel>({
  name: 'question',
  defaults: {
    question: {
      data: [],
      total: 0,
    },
    selectedQuestion: null,
  },
})
@Injectable()
export class QuestionAnswersState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private questionsAnswersService = inject(QuestionsAnswersService);

  @Selector()
  static questionAnswers(state: QuestionAnswersStateModel) {
    return state.question;
  }

  @Selector()
  static selectedQuestionAnswers(state: QuestionAnswersStateModel) {
    return state.selectedQuestion;
  }

  @Action(GetQuestionAnswersAction)
  getQuestionAnswers(
    ctx: StateContext<QuestionAnswersStateModel>,
    action: GetQuestionAnswersAction,
  ) {
    return this.questionsAnswersService.getQuestionAnswers(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            question: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditQuestionAnswersAction)
  edit(ctx: StateContext<QuestionAnswersStateModel>, { id }: EditQuestionAnswersAction) {
    return this.questionsAnswersService.editQuestionAnswers(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedQuestion: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateQuestionAnswersAction)
  update(
    ctx: StateContext<QuestionAnswersStateModel>,
    { payload, id }: UpdateQuestionAnswersAction,
  ) {
    return this.questionsAnswersService.updateQuestionAnswers(id, payload).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const questions = [...state.question.data];
            const index = questions.findIndex(question => question.id === id);
            questions[index] = result;

            ctx.patchState({
              ...state,
              question: {
                data: questions,
                total: state.question.total,
              },
            });
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Q&A Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteQuestionAnswersAction)
  delete(ctx: StateContext<QuestionAnswersStateModel>, { id }: DeleteQuestionAnswersAction) {
    return this.questionsAnswersService.deleteQuestionAnswers(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetQuestionAnswersAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Q&A Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllQuestionAnswersAction)
  deleteAll(ctx: StateContext<QuestionAnswersStateModel>, { ids }: DeleteAllQuestionAnswersAction) {
    return this.questionsAnswersService.deleteAllQuestionAnswers(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetQuestionAnswersAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Q&A Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
