import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import { GetRefundAction, UpdateRefundStatusAction } from '../action/refund.action';
import { UpdateBadgeValueAction } from '../action/sidebar.action';
import { IRefund } from '../interface/refund.interface';
import { NotificationService } from '../services/notification.service';
import { RefundService } from '../services/refund.service';

export class RefundStateModel {
  refund = {
    data: [] as IRefund[],
    total: 0,
  };
}

@State<RefundStateModel>({
  name: 'refund',
  defaults: {
    refund: {
      data: [],
      total: 0,
    },
  },
})
@Injectable()
export class RefundState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private refundService = inject(RefundService);

  @Selector()
  static refund(state: RefundStateModel) {
    return state.refund;
  }

  @Action(GetRefundAction)
  getRefund(ctx: StateContext<RefundStateModel>, action: GetRefundAction) {
    return this.refundService.getRefunds(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            refund: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateRefundStatusAction)
  updateRefundStatus(
    ctx: StateContext<RefundStateModel>,
    { id, status }: UpdateRefundStatusAction,
  ) {
    return this.refundService.updaterefundStatus(id, status).pipe(
      tap({
        next: result => {
          if (typeof result === 'object') {
            const state = ctx.getState();
            const refunds = [...state.refund.data];
            const index = refunds.findIndex(refund => refund.id === id);
            refunds[index] = result;

            ctx.patchState({
              ...state,
              refund: {
                data: refunds,
                total: state.refund.total,
              },
            });
            this.store.dispatch(
              new UpdateBadgeValueAction('/refund', result?.total_pending_refunds),
            );
          }
        },
        complete: () => {
          this.notificationService.showSuccess('Refund Status Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
