import { Injectable, inject } from '@angular/core';

import { Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetReportAction,
  GetReportsFieldsAction,
  GetReportsTypesAction,
} from '../action/reports.action';
import { IReportsCoupon } from '../interface/reports.interface';
import { ReportsService } from '../services/reports.service';

export interface IReportType {
  title: string;
  field_type: string;
}
export class ReportsStateModel {
  reportsTypes: IReportType[];
  reportsFields: Record<string, unknown>;
  reports = {
    data: [] as IReportsCoupon[],
    total: 0,
  };
}

@State<ReportsStateModel>({
  name: 'reports',
  defaults: {
    reportsTypes: [],
    reportsFields: {},
    reports: {
      data: [],
      total: 0,
    },
  },
})
@Injectable()
export class ReportsState {
  private refundService = inject(ReportsService);

  @Selector()
  static reportsTypes(state: ReportsStateModel) {
    return state.reportsTypes.map((res: { title: string; field_type: string }) => {
      return { label: res?.title, value: res?.field_type };
    });
  }

  @Selector()
  static reportsFields(state: ReportsStateModel) {
    return state.reportsFields;
  }

  @Selector()
  static reports(state: ReportsStateModel) {
    return state.reports;
  }

  @Action(GetReportsTypesAction)
  getReportsTypes(ctx: StateContext<ReportsStateModel>, _action: GetReportsTypesAction) {
    return this.refundService.getReportsTypes().pipe(
      tap({
        next: result => {
          ctx.patchState({
            reportsTypes: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(GetReportsFieldsAction)
  getReportsFields(ctx: StateContext<ReportsStateModel>, { field }: GetReportsFieldsAction) {
    return this.refundService.getReportsFields(field).pipe(
      tap({
        next: result => {
          ctx.patchState({
            reportsFields: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(GetReportAction)
  getReports(ctx: StateContext<ReportsStateModel>, action: GetReportAction) {
    return this.refundService.getReports(action.report, action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            reports: {
              data: result.data,
              total: result?.total ? result?.total : result.data?.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
