import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetReviewsAction,
  DeleteReviewAction,
  DeleteAllReviewAction,
} from '../action/review.action';
import { IReview } from '../interface/review.interface';
import { NotificationService } from '../services/notification.service';
import { ReviewService } from '../services/review.service';

export class ReviewStateModel {
  review = {
    data: [] as IReview[],
    total: 0,
  };
}

@State<ReviewStateModel>({
  name: 'review',
  defaults: {
    review: {
      data: [],
      total: 0,
    },
  },
})
@Injectable()
export class ReviewState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private reviewService = inject(ReviewService);

  @Selector()
  static review(state: ReviewStateModel) {
    return state.review;
  }

  @Action(GetReviewsAction)
  getReviews(ctx: StateContext<ReviewStateModel>, action: GetReviewsAction) {
    return this.reviewService.getReviews(action?.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            review: {
              data: result.data,
              total: result?.total ? result?.total : result.data.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteReviewAction)
  delete(ctx: StateContext<ReviewStateModel>, { id }: DeleteReviewAction) {
    return this.reviewService.deleteReview(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetReviewsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Review Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllReviewAction)
  deleteAll(ctx: StateContext<ReviewStateModel>, { ids }: DeleteAllReviewAction) {
    return this.reviewService.deleteAllReview(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetReviewsAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Review Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
