import { Injectable, inject } from '@angular/core';

import { Store, Action, Selector, State, StateContext } from '@ngxs/store';
import { tap } from 'rxjs';

import {
  GetRolesAction,
  GetRoleModulesAction,
  CreateRoleAction,
  EditRoleAction,
  UpdateRoleAction,
  DeleteRoleAction,
  DeleteAllRoleAction,
} from '../action/role.action';
import { IRole, IModule } from '../interface/role.interface';
import { NotificationService } from '../services/notification.service';
import { RoleService } from '../services/role.service';

export class RoleStateModel {
  role = {
    data: [] as IRole[],
    total: 0,
  };
  selectedRole: IRole | null;
  modules: IModule[];
}

@State<RoleStateModel>({
  name: 'role',
  defaults: {
    role: {
      data: [],
      total: 0,
    },
    selectedRole: null,
    modules: [],
  },
})
@Injectable()
export class RoleState {
  private store = inject(Store);
  private notificationService = inject(NotificationService);
  private roleService = inject(RoleService);

  @Selector()
  static role(state: RoleStateModel) {
    return state.role;
  }

  @Selector()
  static roles(state: RoleStateModel) {
    return state.role.data
      .map(res => {
        return { label: res?.name, value: res?.id };
      })
      .filter(value => value.label !== 'admin' && value.label !== 'vendor');
  }

  @Selector()
  static selectedRole(state: RoleStateModel) {
    return state.selectedRole;
  }

  @Selector()
  static roleModules(state: RoleStateModel) {
    return state.modules;
  }

  @Action(GetRolesAction)
  getRoles(ctx: StateContext<RoleStateModel>, action: GetRolesAction) {
    return this.roleService.getRoles(action?.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            role: {
              data: result.data,
              total: result?.total ? result?.total : result.data.length,
            },
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(GetRoleModulesAction)
  getRoleModules(ctx: StateContext<RoleStateModel>) {
    return this.roleService.getRoleModules().pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            modules: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(CreateRoleAction)
  create(ctx: StateContext<RoleStateModel>, action: CreateRoleAction) {
    return this.roleService.createRole(action.payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            role: {
              data: [...state.role.data, result],
              total: state?.role.total + 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Role Created Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(EditRoleAction)
  edit(ctx: StateContext<RoleStateModel>, { id }: EditRoleAction) {
    return this.roleService.editRole(id).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          ctx.patchState({
            ...state,
            selectedRole: result,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateRoleAction)
  update(ctx: StateContext<RoleStateModel>, { payload, id }: UpdateRoleAction) {
    return this.roleService.updateRole(id, payload).pipe(
      tap({
        next: result => {
          const state = ctx.getState();
          const roles = [...state.role.data];
          const index = roles.findIndex(role => role.id == id);
          roles[index] = result;

          ctx.patchState({
            ...state,
            role: {
              data: roles,
              total: state.role.total - 1,
            },
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Role Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteRoleAction)
  delete(ctx: StateContext<RoleStateModel>, { id }: DeleteRoleAction) {
    return this.roleService.deleteRole(id).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetRolesAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Role Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(DeleteAllRoleAction)
  deleteAll(ctx: StateContext<RoleStateModel>, { ids }: DeleteAllRoleAction) {
    return this.roleService.deleteAllRole(ids).pipe(
      tap({
        next: () => {
          this.store.dispatch(new GetRolesAction({ page: 1, paginate: 15 }));
        },
        complete: () => {
          this.notificationService.showSuccess('Role Deleted Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
