import { Injectable, inject } from '@angular/core';

import { Action, Selector, State, StateContext, Store } from '@ngxs/store';
import { tap } from 'rxjs';

import { LocalLanguageAction } from '../action/language.action';
import {
  GetAppSettingOptionAction,
  GetSettingOptionAction,
  TestEmailAction,
  UpdateAppSettingOptionAction,
  UpdateSettingOptionAction,
} from '../action/setting.action';
import { IAppValues, IValues } from '../interface/setting.interface';
import { NotificationService } from '../services/notification.service';
import { SettingService } from '../services/setting.service';

export class SettingStateModel {
  setting: IValues | null;
  appSetting: IAppValues | null;
  localLang: string;
}

@State<SettingStateModel>({
  name: 'setting',
  defaults: {
    setting: null,
    appSetting: null,
    localLang: '',
  },
})
@Injectable()
export class SettingState {
  private settingService = inject(SettingService);
  private store = inject(Store);
  private notificationService = inject(NotificationService);

  @Selector()
  static setting(state: SettingStateModel) {
    return state.setting;
  }

  @Selector()
  static appSetting(state: SettingStateModel) {
    return state.appSetting;
  }

  @Action(GetSettingOptionAction)
  getSettingOptions(ctx: StateContext<SettingStateModel>) {
    return this.settingService.getSettingOption().pipe(
      tap({
        next: result => {
          if (result && result.values) {
            ctx.patchState({
              setting: result.values,
            });

            const localLang = this.store.selectSnapshot(
              state => state && state.language && state.language.localLanguage,
            );
            if (localLang) {
              this.store.dispatch(new LocalLanguageAction(localLang));
            } else {
              this.store.dispatch(
                new LocalLanguageAction(result?.values?.general?.default_language),
              );
            }
          }
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateSettingOptionAction)
  updateSettingOption(ctx: StateContext<SettingStateModel>, action: UpdateSettingOptionAction) {
    return this.settingService.updateSettingOption(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            setting: result.values,
          });
          this.store.dispatch(new LocalLanguageAction(result?.values?.general?.default_language));
        },
        complete: () => {
          this.notificationService.showSuccess('Settings Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(TestEmailAction)
  TestMailSetting(ctx: StateContext<SettingStateModel>, action: TestEmailAction) {
    return this.settingService.testMail(action.payload).pipe(
      tap({
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(GetAppSettingOptionAction)
  getAppSettingOptions(ctx: StateContext<SettingStateModel>) {
    return this.settingService.getAppSettingOption().pipe(
      tap({
        next: result => {
          ctx.patchState({
            appSetting: result.values,
          });
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }

  @Action(UpdateAppSettingOptionAction)
  UpdateAppSettingOptionAction(
    ctx: StateContext<SettingStateModel>,
    action: UpdateAppSettingOptionAction,
  ) {
    return this.settingService.updateAppSettingOption(action.payload).pipe(
      tap({
        next: result => {
          ctx.patchState({
            appSetting: result.values,
          });
        },
        complete: () => {
          this.notificationService.showSuccess('Settings Updated Successfully');
        },
        error: err => {
          throw new Error(err?.error?.message);
        },
      }),
    );
  }
}
